/*
 *      COI 3270 print filter
 *      removes VT   /ESC VT for positive/negative full line feeds,
 *              ESC D/ESC U  for positive/negative half line feeds,
 *              ESC G/ESC 4  for graphics mode on/off,
 *                           graphics mode partial hor/ver motion,
 *              BS           for backspacing
 */

#include <stdio.h>


			    /********************************************/
#define SPACE     040       /* SP                                       */
#define BACK_SPC  010       /* BS                                       */
#define TAB       011       /* HT                                       */
#define CARG_RET  015       /* CR                                       */
#define NEWLINE   012       /* LF becomes CR LF                         */
#define LINEFEED  013       /* VT becomes LF                            */
#define SHIFT_UP  002       /* to simulate subscripts/superscripts      */
#define SHIFT_DN  003       /* to simulate subscripts/superscripts      */
#define ESCAPE    033       /* to simulate negative and half line feeds */
#define NEG_LF    013       /* escaped VT  for negative line feed       */
#define POS_HLF   'U'       /* escaped 'U' for positive half line       */
#define NEG_HLF   'D'       /* escaped 'D' for negative half line feed  */
#define ON        'G'       /* escaped 'G' for graphics mode on         */
#define OFF       '4'       /* escaped '4' for graphics mode off        */
#define V_RES       8       /* vertical resolution per row              */
#define H_RES      10       /* horizontal resolution per column         */
			    /********************************************/
char **page;                /* char array: page[1:length,0:width+1]     */
int written;                /* number of graphic chars on current page  */
int length = 66;            /* page length, default 66 lines/page       */
int width  = 80;            /* page width,  default 80 characters/line  */
int offset =  0;            /* page offset, default  0 characters       */
int row, col;               /* position to place next character         */
int graphics;               /* graphics mode is ON or OFF               */
int shift;                  /* shift up for superscripts                */
int p_row;                  /* 0:7, partial distance above current row  */
int p_col;                  /* 0:9, partial distance right of col       */
int c;                      /* current input character                  */
char *malloc();             /* standard memory allocation routine       */
			    /********************************************/

main(argc, argv)
int argc;
char *argv[];
{
	FILE *fp;

	argc--;
	argv++;

	while (argc && argv[0][0] == '-') {
		switch (argv[0][1]) {
		case 'w':
			width  = atoi(*argv+2);
			break;
		case 'l':
			length = atoi(*argv+2);
			break;
		case 'o':
			offset = atoi(*argv+2);
			break;
		default:
			fprintf(stderr, "tprf:  invalid %s option\n", *argv);
			exit(1);
			break;
		}
		argc--;
		argv++;
	}

	page = (char **) malloc(length*sizeof(char *));
	if (page == NULL) {
		fprintf(stderr, "tprf: out of memory\n");
		exit(1);
	}
	page--;
	for (row=1; row<=length; row++) {
		page[row] = malloc(1+width+1);
		if (page[row] == NULL) {
			fprintf(stderr, "tprf: out of memory\n");
			exit(1);
		}
		page[row][0] = '\0';
		for (col=1; col<=width; col++)
			page[row][col] = ' ';
	}
	row     = 1;
	col     = 1+offset;
	graphics = OFF;
	shift   = 0;
	p_row   = 0;
	p_col   = 0;
	written = 0;

	if (argc) {
		while(argc) {
			fp = fopen(*argv, "r");
			if (fp == NULL) {
				fprintf(stderr, "tprf:  cannot open %s\n", *argv);
				exit(1);
			}
			tprf(fp);
			fclose(fp);
			argv++;
			argc--;
		}
	} else {
		if (isatty(fileno(stdin))) {
			fprintf(stderr, "tprf:  no files specified\n");
			exit(1);
		} else {
			tprf(stdin);
		}
	}

	if (written) {
		row = length;
		linefeed(1);
	}
}

tprf(fp)
FILE *fp;
{
	char *p;

	for (;;) {
		switch(c = getc(fp)) {
		case SPACE:
			if (graphics == ON) {
				p_col++;
				if (p_col >= H_RES) {
					p_col -= H_RES;
					col++;
				}
			} else {
				col++;
			}
			break;
		case BACK_SPC:
			if (graphics == ON) {
				p_col--;
				if (p_col < 0) {
					p_col += H_RES;
					col--;
				}
			} else {
				col--;
			}
			break;
		case TAB:
			c = (col-1) % 8;
			if (c >= 0)
				col += 8;
			col -= c;
			p_col = 0;
			break;
		case CARG_RET:
			col = 1+offset;
			p_col = 0;
			break;
		case NEWLINE:
			col = 1+offset;
			p_col = 0;
			if (graphics == ON)
				linefeed(1);
			else
				linefeed(V_RES);
			break;
		case LINEFEED:
			if (graphics == ON)
				linefeed(1);
			else
				linefeed(V_RES);
			break;
		case ESCAPE:
			switch(c = getc(fp)) {
			case NEG_LF:
				if (graphics == ON)
					linefeed(-1);
				else
					linefeed(-V_RES);
				break;
			case POS_HLF:
				linefeed(V_RES/2);
				break;
			case NEG_HLF:
				linefeed(-V_RES/2);
				break;
			case ON:
			case OFF:
				graphics = c;
				break;
			}
			break;
		case EOF:
			return;
			break;
		case SHIFT_UP:
			shift++;
			break;
		case SHIFT_DN:
			shift--;
			break;
		default:
			if (c > SPACE) {
				if (1 <= col && col <= width && row > 0) {
					p = &page[row][col];
					if (shift>0 && ('0'<=c && c<='9'))
					        c |= 0200;
					if (*p == ' ' || *p == '_')
						*p = c;
				}
				if (graphics == OFF) {
					col++;
				}
				written++;
			}
			break;
		}
	}
}

linefeed(amount)
int amount;
{
	p_row -= amount;
	if (p_row < 0) {
		p_row += V_RES;
		row++;
		if (row > length) {
			output();
			row     = 1;
			written = 0;
		}
	} else if (p_row >= V_RES) {
		p_row -= V_RES;
		row--;
	}
}

output()
{
	int r, c;
	for (r=1; r<=length; r++) {
		for (c=width; page[r][c]==' '; c--) ;
		page[r][c+1] = '\0';
		printf("%s\n", &page[r][1]);
		for (c++; c>=1; c--)
			page[r][c] = ' ';
	}
}
