

/*
 *  Runtime routines for Quickscreen.
 */
#include <stdio.h>
#include <sgtty.h>
#include <signal.h>

#define NOWAIT  04000000
#define MAXOFF  1920
#define MAXCOL  80
#define MAXROW  24
#define MAXINF  200
#define BUFSIZE 3000
#define NULL    0
/*
 * AID chars
 */
#define ENTER 0x7d
#define TSTRQ 0xf0
#define PF1   0xf1
#define PF2   0xf2
#define PF3   0xf3
#define PF4   0xf4
#define PF5   0xf5
#define PF6   0xf6
#define PF7   0xf7
#define PF8   0xf8
#define PF9   0xf9
#define PF10  0x7a
#define PF11  0x7b
#define PF12  0x7c
#define PF13  0xc1
#define PF14  0xc2
#define PF15  0xc3
#define PF16  0xc4
#define PF17  0xc5
#define PF18  0xc6
#define PF19  0xc7
#define PF20  0xc8
#define PF21  0xc9
#define PF22  0x4a
#define PF23  0x4b
#define PF24  0x4c
#define PA1   0x6c
#define PA2   0x6e
#define PA3   0x6b
#define CLEAR 0x6d
/*
 * hex constants for 3270 control chars
 */
#define RDB 0x02
#define WRB 0x01
#define ERW 0x05
#define EAU 0x0f
#define RDM 0x06
#define SF  0x1d
#define IC  0x13
#define SBA 0x11
#define RA  0x3c
#define SOH 0x01
#define STX 0x02
#define RAW 040
/*
 * indices into tubecc array for attribute chars
 */
#define OUT    32
#define BRIGHT  8
#define OFF    12
/*
 * mask to find out if the tube is COI or Memorex
 */
#define ODBL 0400000
/*
 * global declarations that users have access to
 */
int qskp = -1;
int qscrow = 1, qsccol = 1;
int qstrim = 1;
int qsodbl = -1;
int qsalarm = 0;
/*
 * static (local) global declarations
 */
static int qsoflg = 0,
           qsrow = 1,
           qsslm = 1,
           qsso = 0,
           qsplm = 1;
static int lasticmd, lastocmd, lastwcmd;
static int qsfsfd;

static char qstiobf[BUFSIZE], *qstiop;
 
static struct ift {        /* input fields table */
        char *vp;
        char sba[2];
        char modflg;
} qsift[MAXINF], *qsiftp;
/*
 * character translation tables from tubecc.s
 */
extern char tubecc[], tubitab[], tubotab[], mrxotab[];

static char *cpytrexin();
static char *cpytrinex();
 
qsat(rowcol)
int rowcol;
{
        int row, col;

        row = rowcol/100;
        col = rowcol%100;
        qsrow = (row-1)%MAXROW + 1;
        qsslm = (col-1)%MAXCOL + 1;
        qsso = ((qsrow-1)*MAXCOL + qsslm)%MAXOFF;
	qsso = (qsso == 0)? 1919: qsso-1;
}

qsbfic()
{
        *qstiop++ = IC;
}

qsbfinit(eau, erase, bell, lock, resetmdt)
int eau, erase, bell, lock, resetmdt;
{
	char wcc;
        int ocmd;
	struct {
		unsigned          : 5;
		unsigned bell     : 1;
		unsigned unlock   : 1;
		unsigned resetmdt : 1;
	};

	if (!qsoflg)
		if (qsopen() < 0)
			error("cannot open tube\n");
	ocmd =   (eau)? EAU:
               (erase)? ERW:
                        WRB;
        if (ocmd == ERW) {
                qsiftp = qsift;
                qsso = 0;
        }
	if (lastocmd != ocmd) {
		ioctl(qsfsfd, TUBOCMD, ocmd);
		lastocmd = ocmd;
	}
	wcc = 0;
	wcc.bell = (bell)? 1: 0;      /* bell can be an expression */
	wcc.unlock = (lock)? 0: 1;
	wcc.resetmdt = (resetmdt)? 1: 0;
        qstiop = qstiobf;
	*qstiop++ = tubecc[wcc];
}

qsbfsba(rowcol)
int rowcol;
{
        int row, col;

        if (rowcol < 101)
		rowcol=101;
        row = rowcol / 100;
        col = rowcol % 100;
        qsrow = (row-1)%MAXROW+1;
        col = (col-1)%MAXCOL+1;
        qsso = ((qsrow-1)*MAXCOL + col)%MAXOFF;
	qsso = (qsso == 0)? 1919: qsso-1;
        *qstiop++ = SBA;
        *qstiop++ = tubecc[qsso >> 6];
        *qstiop++ = tubecc[qsso & 077];
}

qstxtinit(plm)
int plm;
{
        qsplm=plm;
}

qsbfra()
{
        qsbflinit(0, qsplm);
        *qstiop++ = RA;
        qsso = (qsso+MAXCOL-(qsso%MAXCOL))%MAXOFF;
        *qstiop++ = tubecc[qsso>>6];
        *qstiop++ = tubecc[qsso & 077];
        *qstiop++ = NULL;
}

qsbflinit(skips, bcol)
int skips, bcol;
{
        int col;
        qsrow = (qsrow+skips-1)%MAXROW + 1;
        col = qsslm + (bcol - qsplm);
        qsso = (MAXCOL*(qsrow-1) + col)%MAXOFF;
	qsso = (qsso == 0)? 1919: qsso-1;
        *qstiop++ = SBA;
        *qstiop++ = tubecc[qsso>>6];
        *qstiop++ = tubecc[qsso & 077];
        qsrow = (qsrow==24) ? 1 : qsrow+1;
}

qsbuffld(attr, fp, fl)
int attr, fl;
char *fp;
{
	int type;
        char *p;
	char buf[20];

        if (attr < 0 || 9 < attr)
                error("illegal attribute: %d\n", attr);
	if (attr != 0) {
	        *qstiop++ = SF;
	        type = 0;
	        if (1 <= attr && attr <= 3)
		        type += OUT;
	        if (attr%3 == 2)
		        type += BRIGHT;
	        else if (attr%3 == 0)
		        type = OFF;
	        *qstiop++ = tubecc[type];
	        qsso = ++qsso%MAXOFF;
                if (attr > 3) {
                        if (qsiftp >= &qsift[MAXINF])
                                error("maximum input fields exceeded\n");
                        qsiftp->sba[0] = tubecc[qsso>>6];
                        qsiftp->sba[1] = tubecc[qsso & 077];
                        qsiftp->vp = fp;
			++qsiftp;
                        if (attr > 6)
                                *qstiop++ = IC;
                }
        }
        if (fl <= 0) {
                fl = 0;
                p = fp;
                while (*p++)
			++fl;
        }
	qstiop = cpytrinex(qstiop, fp, qstiop+fl);
        qsso = (qsso+fl)%MAXOFF;
}

copyscreen(bp)
char *bp;
{
        char *p;
        int n;

	if (!qsoflg)
		if (qsopen() < 0)
			error("cannot open tube\n");
	if (lasticmd != RDB) {
		ioctl(qsfsfd, TUBICMD, RDB);
		lasticmd = RDB;
	}
	if (lastwcmd != TUBNATTN) {
		ioctl(qsfsfd, TUBNATTN, 0);
		lastwcmd = TUBNATTN;
	}
        n = read(qsfsfd, qstiobf, BUFSIZE);
        p = qstiobf+3;
        while (p < qstiobf+n)
                if (!*p) {
			*bp++ = ' ';
			++p;
                } else if (*p == SF) {
                        *bp++ = ' ';
                        p += 2;
		} else
	                *bp++ = tubitab[*p++];
	*bp = 0;
}
 
qswrite()
{
int i;
        struct ift *ip;

        for (ip = qsift; ip < qsiftp; ip++)
                ip->modflg = 0;
        write(qsfsfd, qstiobf, qstiop-qstiobf);
}

qsread(waitforaid)
int waitforaid;
{
        int wcmd, n, tmp;
        struct ift *ip;

	if (lasticmd != RDM) {
		ioctl(qsfsfd, TUBICMD, RDM);
		lasticmd = RDM;
	}
	wcmd = (waitforaid)? TUBWATTN: TUBNATTN;
	if (lastwcmd != wcmd) {
		ioctl(qsfsfd, wcmd, 0);
		lastwcmd = wcmd;
	}
        n = read(qsfsfd, qstiobf, BUFSIZE);
        if (n == 0)
		return(0);
        switch(qstiobf[0]) {      /* AID char */
        case ENTER:
                qskp = 0;
                break;
        case TSTRQ:
                /*
                 * this only happens during copyscreen
                 */
                qskp = 29;
                break;
        case PF1:  case PF2:  case PF3:  case PF4:  case PF5:
        case PF6:  case PF7:  case PF8:  case PF9:  case PF10:
        case PF11: case PF12:
                qskp = qstiobf[0] & 017;
                break;
        case PF13: case PF14: case PF15: case PF16: case PF17:
        case PF18: case PF19: case PF20: case PF21: case PF22:
        case PF23: case PF24:
                qskp = (qstiobf[0] & 017) + 12;
                break;
        case PA1:
                qskp = 25;
                return(n);
        case PA2:
                qskp = 26;
                return(n);
        case PA3:
                qskp = 27;
                return(n);
        case CLEAR:
                qskp = 28;
                qsrow = 1;
                qsso = 0;
                qsiftp = qsift;
                return(n);
        case SOH:  /* TEST REQ */
                if (qstiobf[1]==tubotab['%'] &&
                    qstiobf[2]==tubotab['/'] &&
                    qstiobf[3]==STX) {
                        qskp = 29;
                        break;
                }
        default:
                qskp = 100;
                return(n);
        }
	/*
         * we have a valid key with data
	 */
        if (qskp != 29 || qstiobf[0] == TSTRQ) {
                qstiop = &qstiobf[3];
                tmp = ((qstiobf[1] & 077)<<6) +
                        (qstiobf[2] & 077);
                qscrow = tmp/MAXCOL + 1;
                qsccol = tmp%MAXCOL + 1;
        } else
                qstiop = &qstiobf[4];
	/*
         * now we move data out of the buffer into the variables
	 */
        while (qstiop < qstiobf + n) {
                while (qstiop < qstiobf+n && *qstiop++ != SBA)
			;
                if (qstiop == qstiobf+n)
                        break;
                ip = qsift;
		/*
                 * search ift table for variable with same SBA
                 */
                while (ip < qsiftp) {
                        if (ip->sba[0] == *qstiop &&
                            ip->sba[1] == *(qstiop+1))
                                break;
                        ip++;
                }
                if (ip == qsiftp)
			/*
                         * no corresponding entry found so we
                         * skip to next input field
			 */
                        continue;
                else {
			/*
                         * copy input field to variable
			 */
                        qstiop += 2;        /* skip SBA bytes */
                        qstiop = cpytrexin(ip->vp, qstiop, qstiobf+n);
                        ip->modflg = 1;
                }
        }
        return(n);
}

/*
 * "copy and translate internal to external"
 * Copy t to s until you see a null byte in t or until s >= last.
 * t is in ASCII and s should be in EBCDIC.
 * If qsodbl is true then
 * certain chars from t should become 2 char sequences in s.
 * They are { becomes ( <, [ becomes ( |, } becomes > ), ] becomes | ),
 * ^ becomes \ ~, and ` becomes \ <.
 * Fill up s with null bytes until s >= last.
 * Return where s finally ended up.
 */

static
char *
cpytrinex(s, t, last)
char *s, *t, *last;
{
	if (!qsodbl)
                while (*t && s < last)
                       *s++ = mrxotab[*t++];
	else {
		while (*t && s < last) {
			switch (*t) {
			case '{':
				*s++ = tubotab['('];
				*s++ = tubotab['<'];
				break;
			case '[':
				*s++ = tubotab['('];
				*s++ = tubotab['|'];
				break;
			case '}':
				*s++ = tubotab['>'];
				*s++ = tubotab[')'];
				break;
			case ']':
				*s++ = tubotab['|'];
				*s++ = tubotab[')'];
				break;
			case '^':
				*s++ = tubotab['\\'];
				*s++ = tubotab['~'];
				break;
			case '`':
				*s++ = tubotab['\\'];
				*s++ = tubotab['<'];
				break;
			default:
				*s++ = tubotab[*t];
				break;
			}
			++t;
	        }
	}
        while (s < last)
                *s++ = NULL;
	return(s);
}

/*
 * "copy and translate external to internal"
 * Copy t to s until you see an SBA byte in t or until
 * t >= last.
 * t is in EBCDIC and s should be in ASCII.
 * Certain sequences of 2 chars should become 1 in s.
 * They are: ( < becomes {, ( | becomes [, > ) becomes }, | ) becomes ],
 * \ ~ becomes ^, and \ < becomes `.
 * After copying, translating, and collapsing, if 'qstrim' is non-zero
 * trim trailing blanks from s.
 * Put a null byte at the end of s and return where t finally ended up.
 */

static
char *
cpytrexin(s, t, last)
char *s, *t, *last;
{
	char c;
	char *os, *prev;

	os= s;
	prev = "";
	while (t < last && *t != SBA) {
		c = tubitab[*t++];        /* EBCDIC to ASCII */
		switch (*prev) {
		case '(':
			if (c == '<')
				*prev = '{';
			else if (c == '|')
				*prev = '[';
			else
				*s++ = c;
			break;
		case '>':
			if (c == ')')
				*prev = '}';
			else
				*s++ = c;
			break;
		case '|':
			if (c == ')')
				*prev = ']';
			else
				*s++ = c;
			break;
		case '\\':
			if (c == '~')
				*prev = '^';
			else if (c == '<')
				*prev = '`';
			else
				*s++ = c;
			break;
		default:
			*s++ = c;
			break;
		}
		prev = s-1;
	}
	if (qstrim) {
		while (--s >= os && *s == ' ')
			;
		++s;
	}
	*s = NULL;
	return(t);
}

int
qsmod(s)
char *s;
{
        struct ift *ip;

        for (ip = qsift; ip < qsiftp; ++ip)
                if (ip->vp == s)
                        return(ip->modflg);
        return(0);
}

int
qspos(s)
char *s;
{
        int so, row, col;
	struct ift *ip;
 
        for (ip = qsift; ip < qsiftp; ++ip)
                if (ip->vp == s) {
                        so = ((ip->sba[0] & 077) << 6) +
                              (ip->sba[1] & 077);
			row = so/MAXCOL + 1;
			col = so%MAXCOL + 1;
                        return(row*100+col);
                }
        return(101);
}

/*
 * qssleep is needed in case the sleep expression
 * evaluates to 0.  A regular sleep would cause an interrupt
 * and make the screen blink between the qswrite and the qsread.
 */

qssleep(nsecs)
int nsecs;
{
	if (nsecs > 0)
		sleep(nsecs);
}

qscatchalarm()
{
	signal(SIGALRM, SIG_IGN);
	qsalarm = 1;
}

qsopen()
{
	char *p;
	struct sgttyb buf;

	if (qsoflg)
		return(0);
        if ((p = ttyname(2)) == 0 &&
            (p = ttyname(1)) == 0 &&
            (p = ttyname(0)) == 0   )
		return(-1);
	/*
	 *  01234567890
	 *  /dev/ttynnn -> /dev/tubnnn
	 */
        p[5] = 't';
        p[6] = 'u';
        p[7] = 'b';
        if ((qsfsfd = open(p, 2)) < 0)
                return(-1);
        if (qsodbl == -1) {
	        if (gtty(2, &buf) < 0 &&
	            gtty(1, &buf) < 0 &&
	            gtty(0, &buf) < 0   )
                        error("cannot do gtty\n");
                qsodbl = (buf.sg_flags & ODBL);
	}
	lasticmd = RDM;
	lastocmd = ERW;
	lastwcmd = TUBWATTN;
        ++qsoflg;
}

qsclose()
{
	close(qsfsfd);
	qsoflg = 0;
}

static
error(args)
int *args;
{
	if (qsoflg)
	        close(qsfsfd);
	fprintf(stderr, "qs: %r", &args);
	exit(1);
}
