#include "../h/conf.h"
#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/io.h"
#include "../h/tty.h"
 
/*
 * driver for UNIX/470 virtual printer
 */
 
extern char atetab[];

#define PRPRI   PUSER
#define PRMAX    1                         /* number of printers       */
#define BUFFSIZE 156                       /* buffer size              */
#define MAXCHAIN 24                        /* number of buffers (even) */
#define WCR      0x09                      /* write with newline       */
#define WNCR     0x01                      /* write with no nl         */
#define WFF      0x89                      /* form feed                */
#define TIC      0x08                      /* transfer in channel */
#define LBRACE   0x8b
#define RBRACE   0x9b
#define NOTSIGN  0x5f
#define CENTSIGN 0x4a
#define GA       0140
 
/*
 * states
 */
 
#define FREE  0
#define OPEN  1
#define DONE  2
#define ERROR 3
 
/*
 * info structure
 */
 
struct vmprinter {
        ccw_t  pr_ccws[MAXCHAIN+1];
        char pr_buff[MAXCHAIN][BUFFSIZE];     /* buffers             */
	int  pr_bufuse[MAXCHAIN];             /* buffer use flags    */
	int  pr_ficp;                         /* First In Chan Prog  */
	int  pr_avail;                        /* is printer available*/
        int  pr_line;                         /* current buffer      */
	int  pr_offset;                       /* current position    */
        int  pr_addr;                         /* address for caw     */
        char pr_state;                        /* state of printer    */
        char pr_raw;                          /* print mode          */
        char pr_more;                         /* ready for more      */
} pr_info[PRMAX];
 
/*
 * NEXT(x) = next x in circular chain.
 * PREV(x) = previous x in circular chain.
 */
#define NEXT(Z) ( (Z==MAXCHAIN-1) ? 0 : Z+1 )
#define PREV(Z) ( (Z==0) ? MAXCHAIN-1 : Z-1 )
#define SPACE  atetab[' ']
#define SET 1
#define CLEAR 0
#define NUL 0
#define SO 016
#define SI 017
#define RETURN   015
#define FORMFEED 014

/*
 * Alternate character translation table for TN print train.
 * Selected when high-order bit of output character is on.
 */
char alttab[128] = {
        0x9C,   0x9E,   0x9F,   0xA1,   /* 200-203 */
        0xAB,   0xAC,   0xBB,   0xBC,   /* 204-207 */
        0, 0, 0, 0, 0, 0, 0, 0,         /* 210-217 */
        0, 0, 0, 0, 0, 0, 0, 0,         /* 220-227 */
        0, 0, 0, 0, 0, 0, 0, 0,         /* 230-237 */
        0, 0, 0, 0, 0, 0, 0, 0,         /* 240-247 */
        0x8D,   0x9D,   0xAF,   0x8E,   /* 250-253 */
        0,      0xBF,   0,      0,      /* 254-257 */
        0xB0,   0xB1,   0xB2,   0xB3,   /* 260-263 */
        0xB4,   0xB5,   0xB6,   0xB7,   /* 264-267 */
        0xB8,   0xB9,   0,      0,      /* 270-273 */
        0x8C,   0xBE,   0xAE,   0,      /* 274-277 */
        0, 0, 0, 0, 0, 0, 0, 0,         /* 300-307 */
        0, 0, 0, 0, 0, 0, 0, 0,         /* 310-317 */
        0, 0, 0, 0, 0, 0, 0, 0,         /* 320-327 */
        0, 0, 0, 0, 0, 0, 0, 0xA0,      /* 330-337 */
        0, 0, 0, 0, 0, 0, 0, 0,         /* 340-347 */
        0, 0, 0, 0, 0, 0, 0, 0,         /* 350-357 */
        0, 0, 0, 0, 0, 0, 0, 0,         /* 360-367 */
        0, 0, 0, 0, 0, 0, 0, 0,         /* 370-377 */
};

/*
 * Set initial states, set up ccws, clear buffers.
 */
/* ARGSUSED */
propen(dev,flag)
int  dev, flag;
{
        int row, col;
        struct vmprinter *pr;

        pr = &pr_info[minor(dev)];
        if(pr->pr_avail != FREE) {
                u.u_error = EBUSY;
                return;
        }
	pr->pr_ficp = 0;
	pr->pr_line = 0;
	pr->pr_offset = 0;
	pr->pr_state = OPEN;
        pr->pr_avail = OPEN;
        pr->pr_raw = CLEAR;
        pr->pr_more = CLEAR;
        pr->pr_addr = cdevsw[major(dev)].d_addrs[minor(dev)];
	/*
	 * Last ccw is transfer to first, the rest refer to buffer of
	 * the same subscript.
	 */
	pr->pr_ccws[MAXCHAIN].cc_cmd = TIC;
	pr->pr_ccws[MAXCHAIN].cc_addr = (int)&pr->pr_ccws[0];
	for(row=0; row<MAXCHAIN; row++) {
		pr->pr_ccws[row].cc_addr = (int)pr->pr_buff[row];
		pr->pr_ccws[row].cc_sli  = 1;
		pr->pr_ccws[row].cc_count  = BUFFSIZE;
		pr->pr_bufuse[row] = 0;
		for(col=0; col<BUFFSIZE; col++)
			pr->pr_buff[row][col] = SPACE;
	}
}
 
/*
 * free printer, flushing buffers if needed.
 */
prclose(dev)
int dev;
{
        struct vmprinter *pr;

        pr = &pr_info[minor(dev)];
        if (pr->pr_more == SET) {
	        prnext(pr, WCR, CLEAR);
	        issuesio(pr);
	}
        pr->pr_avail = FREE;
        pr->pr_state = FREE;
        cpclose(pr->pr_addr);
}
 
/*
 * Fill buffers.
 */
prwrite(dev)
{
        int c;
        struct vmprinter *pr;

        pr = &pr_info[minor(dev)];
        while ((c = cpass()) != -1 && u.u_error != EIO){
                pr->pr_more = SET;
                if(pr->pr_raw) {
                        pr->pr_buff[pr->pr_line][pr->pr_offset] = c;
                        if(++pr->pr_offset >= BUFFSIZE)
                                prnext(pr, WCR, CLEAR);
                        continue;
                }
                switch(c){
                        case NUL:
                        case SO:
                        case SI:
                                break;

                        case '\t':
                                pr->pr_offset++;
                                while (pr->pr_offset % 8){
                                        if (pr->pr_offset >= BUFFSIZE)
                                                prnext(pr, WCR, CLEAR);
                                        pr->pr_offset++;
                                }
                                break;
                        case '\n':
                                prnext(pr, WCR, CLEAR);
                                break;
                        case RETURN:
                                pr->pr_offset = 0;
                                break;
                        case FORMFEED:
                                prnext(pr, WFF, CLEAR);
                                break;
                        case '\b':
                                if (pr->pr_offset)
                                        --pr->pr_offset;
                                break;
                        case '{':
                                c = LBRACE; goto norm;
                        case '}':
                                c = RBRACE; goto norm;
                        case '~':
                                c = NOTSIGN; goto norm;
                        case '\\':
                                c = CENTSIGN; goto norm;
			case GA:
				c = '\'';
				/* fall through */
                        default:
                                if(c & 0200)
                                        c = alttab[c & 0177];
                                else
                                        c = atetab[c];
                norm:           if (pr->pr_buff[pr->pr_line][pr->pr_offset] != SPACE)
                                        nextfill(pr, c);
				else
					pr->pr_buff[pr->pr_line][pr->pr_offset] = c;
                                pr->pr_offset++;
                                if (pr->pr_offset >= BUFFSIZE)
                                        prnext(pr, WCR, CLEAR);
                        }
                }
                if(pr->pr_raw)
                        prnext(pr, WCR, CLEAR);
        }
 
/*
 * Issue sio instruction, unchain ccws, clear buffers.
 */
issuesio(pr)
struct vmprinter *pr;
{
        int printr(), row, col;

        sio(pr->pr_addr, &pr->pr_ccws[pr->pr_ficp], printr, (int)pr);
        while(pr->pr_state == OPEN)
                sleep((caddr_t)pr, PRPRI);
        if(pr->pr_state == ERROR)
                u.u_error = EIO;
        pr->pr_state = OPEN;
	/*
	 * Clear buffers, unchain ccws, reset front of channel program.
	 */
	row = pr->pr_ficp;
	while(pr->pr_ccws[row].cc_cc) {
		for(col=0; col<BUFFSIZE; col++)
			pr->pr_buff[row][col] = SPACE;
		pr->pr_ccws[row].cc_cc = 0;
		row = NEXT(row);
	}
	for(col=0; col<BUFFSIZE; col++)
		pr->pr_buff[row][col] = SPACE;
	pr->pr_ficp = NEXT(row);
	/*
	 * If buffers are empty.
	 */
	if(pr->pr_ficp == pr->pr_line && pr->pr_offset == 0)
	        pr->pr_more = CLEAR;
	return;
}
 
/*
 * Interrupt routine.
 */
/* ARGSUSED */
printr(pr, csw, sense)
struct vmprinter *pr;
csw_t *csw;
char *sense;
{
        if(csw->cs_uc || csw->cs_ue)
                pr->pr_state = ERROR;
        else if(csw->cs_de)
                pr->pr_state = DONE;
        if(pr->pr_state != OPEN)
                wakeup((caddr_t)pr);
}
 
#define RAW 040
 
/* ARGSUSED */
prioctl(dev, cmd, argp, flag)
int dev, cmd, flag;
caddr_t argp;
{
        struct vmprinter *pr;

        pr = &pr_info[minor(dev)];

	switch(cmd) {

	case PIORAW:
		pr->pr_raw = 1;
		break;

	case PIOCOOK:
		pr->pr_raw = 0;
		break;

	case PIOFLUSH:
		if(pr->pr_more == SET) {
			issuesio(pr);
		}
		break;

	default:
		u.u_error = ENOTTY;
		break;
	}
}


/*
 * Add this buffer to the channel program. If the buffers are
 * all in use start the I/O. This routine takes into account
 * the overprinting hassle.
 */
prnext(pr, key, reoffset)
struct vmprinter *pr;
int key, reoffset;
{
	int     overp,
		row,
		firstrow,
		crow;
	overp = 0;

	firstrow = pr->pr_line;
	/*
	 * Set up ccws for any overprinting lines.
	 */
	row = NEXT(firstrow);
	while(pr->pr_bufuse[row]) {
		pr->pr_ccws[PREV(row)].cc_cmd = WNCR;
		pr->pr_bufuse[row] = 0;
		row = NEXT(row);
		overp++;
	}
	if(overp) {
		/*
		 * Chain ccws of overprinted lines.
		 */
		pr->pr_ccws[PREV(row)].cc_cmd = key;
		pr->pr_line = row;
		crow = firstrow;
		while( crow != PREV(row) ) {
			pr->pr_ccws[crow].cc_cc = 1;
			crow = NEXT(crow);
		}
		crow = NEXT(firstrow);
		while(pr->pr_bufuse[crow]) {
			pr->pr_bufuse[crow] = 0;
			crow = NEXT(crow);
		}
	}
	else {
		pr->pr_line = NEXT(pr->pr_line);
		pr->pr_ccws[firstrow].cc_cmd = key;
	}
	pr->pr_ccws[PREV(pr->pr_line)].cc_cc = 0;
	if(firstrow != pr->pr_ficp)
		/*
		 * Chain this line to channel program.
		 */
		pr->pr_ccws[PREV(firstrow)].cc_cc = 1;
	if(pr->pr_line == pr->pr_ficp)
		issuesio(pr);
	if(reoffset == CLEAR)
	        pr->pr_offset = 0;
	return;
}

/*
 * In case of character 'collision', i.e. overprinting.
 */
nextfill(pr, c)
struct vmprinter *pr;
int  c;
{
	int  to,
	     offset;

	offset = pr->pr_offset;
	to = pr->pr_line;
	/*
	 * Repeat until clear spot in overprint line found.
	 */
	do {
		to = NEXT(to);
		if(to == pr->pr_ficp) {
			if(pr->pr_ficp == pr->pr_line) {
				/*
				 * More overprinting than buffers.
				 */
				prnext(pr, WNCR, SET);
			}
			else
			        issuesio(pr);
		}
	} while(pr->pr_buff[to][offset] != SPACE);
	pr->pr_bufuse[to] = 1;
	pr->pr_buff[to][offset] = c;
	return;
}
