/*
 * Raw Mode Driver for DASDs with RPS.
 */

#include "../h/conf.h"
#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/io.h"
#include "../h/ioconf.h"

/*
 * Channel commands
 */
#define SEEK    0x07
#define RDHA    0x1a
#define SETSCTR 0x23
#define SRID    0x31
#define TIC     0x08
#define RDDATA  0x06
#define RDKD    0x0e
#define RDCNT   0x12
#define WRTDATA 0x05
#define WRTCKD  0x1d
#define WRTKD   0x0d
#define MTON    0x80
#define MT      01
#define HIEQ    02
#define HIEQON  0x40
#define SRKY    0x29
 
/*
 * Sense bits
 */
#define stest(byte,bit) (sense[byte]&(1<<(7-bit)))
#define CMDREJ	stest(0,0)
#define INTREQ	stest(0,1)
#define BUSOUT	stest(0,2)
#define EQCHK	stest(0,3)
#define DATACHK	stest(0,4)
#define OVERRUN	stest(0,5)
#define PERMERR	stest(1,0)
#define ENVDATA	stest(2,3)
#define ENDOCYL stest(1,2)
#define NOREC   stest(1,4)
 
#define PRIRIO (PZERO-1)        /* read/write priority */
 
struct rdasd {
	ccw_t   rd_ccw[8];              /* channel program */
	caddr_t rd_idaw[33];            /* idaw list */
	char    rd_bchr[7];             /* disk address */
	char    rd_ser;                 /* serialization flag */
	dev_t   rd_dev;                 /* UNIX device number */
	short   rd_addr;                /* device address */
	short   rd_cc;                  /* TIO condition code */
	short   rd_errcnt;              /* error count */
	short   rd_ioerr;               /* error number */
	int     rd_resid;               /* residual count */
	char    rd_habuf[5];            /* for read HA, never looked at */
	char    rd_key[255];            /* storage for possible key */
} rdasds[NRDASD];

struct rwdreq {
        char *buff;
        char *cchhr;
        char *key;
        char flag;
};

/*
 * Open a DASD device, raw mode.
 */
rwdopen(dev, flag)
{
        int rwdtioi();
        struct rdasd *rd;

	if(minor(dev) >= NRDASD) {
		u.u_error = ENXIO;
		return;
	}
	rd = &rdasds[minor(dev)];
	rd->rd_dev = dev;
        rd->rd_addr = cdevsw[major(dev)].d_addrs[minor(dev)];
	if (rd->rd_addr == 0) {
		u.u_error = ENXIO;
		return;
	}

        /* do tio on device address */
        rd->rd_cc = -1;
        tio(rd->rd_addr, rwdtioi, (int)rd);
        while(rd->rd_cc == -1)
		sleep((caddr_t)&rd->rd_cc, PRIRIO);
        if(rd->rd_cc != 0) {
                u.u_error = ENXIO;
                return;
        }
        /* test writeability, if they want it */
        if(flag && !d_rwdasd(rd->rd_addr)) {
                u.u_error = EROFS;
                return;
        }
}

/*
 * Close RAW DASD
 */
rwdclose(dev)
{
	rdasds[minor(dev)].rd_ser = 0;        /* just in case */
}

/*
 * Read from a disk, Raw mode
 */
rwdread(dev)
{
        struct rwdreq rdreq;
        int i;
	struct rdasd *rd;

	rd = &rdasds[minor(dev)];
        /* serialize */
        while(rd->rd_ser)
                sleep((caddr_t)&rd->rd_ser, PRIRIO);
        rd->rd_ser = 1;

	/* Build common part of read channel programs */
	for(i=0; i<8; i++)
		rd->rd_ccw[i].cc_dblw = 0;
	rd->rd_ccw[0].cc_cmd = SEEK;
	rd->rd_ccw[0].cc_addr = (int)&rd->rd_bchr[0];
	rd->rd_ccw[0].cc_cc = 1;
	rd->rd_ccw[0].cc_count = 6;
	rd->rd_ccw[1].cc_cmd = RDHA;
	rd->rd_ccw[1].cc_addr = (int)&rd->rd_habuf[0];
	rd->rd_ccw[1].cc_cc = 1;
	rd->rd_ccw[1].cc_count = 5;
	rd->rd_ccw[3].cc_cmd = TIC;
	rd->rd_ccw[3].cc_addr = (int)&rd->rd_ccw[2];

	/* Copy user info, and get length of the key */
        i = rwdcopy(&rdreq, rd->rd_bchr+2, rd->rd_key);
	if (i){

		/* we have a non null key, so set up key search chan pgm */

		rd->rd_ccw[2].cc_cmd = SRKY;
		rd->rd_ccw[2].cc_addr = (int)&rd->rd_key[0];
		rd->rd_ccw[2].cc_cc = 1;
		rd->rd_ccw[2].cc_count = i;
                if (rdreq.flag & HIEQ)
                        rd->rd_ccw[2].cc_cmd |= HIEQON;
                if (rdreq.flag & MT)
                        rd->rd_ccw[2].cc_cmd |= MTON;
		rd->rd_ccw[4].cc_cmd = RDDATA;
		rd->rd_ccw[4].cc_addr = (int)&rd->rd_idaw[0];
		rd->rd_ccw[4].cc_sli = 1;
		rd->rd_ccw[4].cc_ida = 1;
		rd->rd_ccw[4].cc_count = u.u_count;

        } else {

                /* cchhr read */

		rd->rd_ccw[2].cc_cmd = SRID;
		rd->rd_ccw[2].cc_addr = (int)&rd->rd_bchr[2];
		rd->rd_ccw[2].cc_cc = 1;
		rd->rd_ccw[2].cc_count = 5;
                if (rdreq.flag & MT)
                        rd->rd_ccw[2].cc_cmd |= MTON;
		rd->rd_ccw[4].cc_cmd = RDKD;
		rd->rd_ccw[4].cc_addr = (int)&rd->rd_idaw[0];
		rd->rd_ccw[4].cc_ida = 1;
		rd->rd_ccw[4].cc_sli = 1;
		rd->rd_ccw[4].cc_count = u.u_count;
        }
        makidaw(0, rd->rd_idaw, rdreq.buff, u.u_count);
	if(!u.u_error)
                rwdio(rd);
	rd->rd_ser = 0;
	wakeup((caddr_t)&rd->rd_ser);
}

/*
 * Raw DASD write
 */
rwdwrite(dev)
{
        struct rwdreq wrreq;
        struct rdasd *rd;
        char count[8];
        int keyl,i;
	ccw_t *cp;

	rd = &rdasds[minor(dev)];
        /* serialize */
        while(rd->rd_ser)
                sleep((caddr_t)&rd->rd_ser, PRIRIO);
        rd->rd_ser = 1;

        keyl = rwdcopy(&wrreq, rd->rd_bchr+2, rd->rd_key);
        makidaw(1, rd->rd_idaw, wrreq.buff, u.u_count);
        if(u.u_error)
		return;

	/* Build common write channel program */
	for(i=0;i<8;i++)
		rd->rd_ccw[i].cc_dblw = 0;
	rd->rd_ccw[0].cc_cmd = SEEK;
	rd->rd_ccw[0].cc_addr = (int)&rd->rd_bchr[0];
	rd->rd_ccw[0].cc_cc = 1;
	rd->rd_ccw[0].cc_count = 6;
	rd->rd_ccw[1].cc_cmd = RDHA;
	rd->rd_ccw[1].cc_addr = (int)&rd->rd_habuf[0];
	rd->rd_ccw[1].cc_cc = 1;
	rd->rd_ccw[1].cc_count = 5;
	rd->rd_ccw[2].cc_cmd = SRID;
	rd->rd_ccw[2].cc_addr = (int)&rd->rd_bchr[2];
	rd->rd_ccw[2].cc_cc = 1;
	rd->rd_ccw[2].cc_count = 5;
	rd->rd_ccw[3].cc_cmd = TIC;
	rd->rd_ccw[3].cc_addr = (int)&rd->rd_ccw[2];

        cp = &rd->rd_ccw[4];
        if (wrreq.flag > 1){
                for (i=0;i<5;i++)
                        count[i] = rd->rd_bchr[i+2];
                rd->rd_bchr[6]--;
                count[5] = keyl;
                count[6] = (u.u_count >> 8) & 0377;
                count[7] = u.u_count & 0377;
		cp->cc_cmd = WRTCKD;
		cp->cc_addr = (int)count;
		cp->cc_cd = 1;
		cp->cc_count = 8;
		cp++;
        }
        if (wrreq.flag && keyl){
		cp->cc_cmd = WRTKD;
		cp->cc_addr = (int)rd->rd_key;
		cp->cc_cd = 1;
		cp->cc_count = keyl;
		cp++;
        }
	cp->cc_cmd = WRTDATA;
	cp->cc_addr = (int)rd->rd_idaw;
	cp->cc_ida = 1;
	cp->cc_count = u.u_count;
        rwdio(rd);
	rd->rd_ser = 0;
	wakeup((caddr_t)&rd->rd_ser);
}

/*
 * Do a raw DASD I/O operation
 */
static rwdio(rd)
struct rdasd *rd;
{
        int rwdintr();

        rd->rd_resid = -1;
        rd->rd_ioerr = rd->rd_errcnt = 0;
        sio(rd->rd_addr, rd->rd_ccw, rwdintr, (int)rd);
	while(rd->rd_ioerr == 0 && rd->rd_resid == -1)
                sleep((caddr_t)rd, PRIRIO);
        if (rd->rd_ioerr)
                u.u_error = rd->rd_ioerr;
        else
                u.u_count = rd->rd_resid;
}

/*
 * DASD interrupt routine
 */
rwdintr(rd, csw, sense)
struct rdasd *rd;
csw_t *csw;
char *sense;
{
        int i, retry,rwdaint();
 
        if(csw->cs_uc) {
                if (NOREC || ENDOCYL){
                        rd->rd_ioerr = ENXIO;
                        wakeup((caddr_t)rd);
                        return;
                }
                prdev("unit check", rd->rd_dev);
		printf("sense=");
		for(i=0;i<24;i++) printf("%2x", sense[i]);
		printf("\n");
		if(INTREQ) {
                        printf("Intervention required on %3x\n", rd->rd_addr);
                        setax(rd->rd_addr, rwdaint, (int)rd);
                        return;
                }
		retry = 0;
		if((BUSOUT || ENVDATA) && !PERMERR) {
                        if(rd->rd_errcnt++ == 0) retry = 1;
		} else if((EQCHK || DATACHK || OVERRUN) && !PERMERR) {
                        if(rd->rd_errcnt++ < 10) retry = 1;
		}
		if(retry) {
                        sio(rd->rd_addr, rd->rd_ccw, rwdintr, (int)rd);
			return;
		}
                rd->rd_ioerr = EIO;
                wakeup((caddr_t)rd);
                return;
        }
        if(csw->cs_de){
                rd->rd_resid = csw->cs_count;
                rd->rd_errcnt = rd->rd_ioerr = 0;
                wakeup((caddr_t)rd);
        }
}
 
/*
 * Check results of tio
 */
/* ARGSUSED */
rwdtioi(rd, csw, sense)
struct rdasd *rd;
csw_t *csw;
char *sense;
{
	rd->rd_cc = csw->cs_cc;
	wakeup((caddr_t)&rd->rd_cc);
}

/*
 * Asynchronous interrupt routine for raw DASDs
 */
/* ARGSUSED */
rwdaint(rd, csw, sense)
struct rdasd *rd;
csw_t *csw;
char *sense;
{
	setax(rd->rd_addr, (int (*)())0, 0);
        if (csw->cs_de)
                sio(rd->rd_addr, rd->rd_ccw, rwdintr, (int)rd);
}

/*
 * Copy in the user's read or write request
 */
rwdcopy(rwdreqp,cchhr,key)
struct rwdreq *rwdreqp;
char *cchhr,*key;
{
	int i;
        register char *p1,*p2;

	/* Copy user rwdreq structure, pointed to by u.u_base */

        copyin(u.u_base, (caddr_t)rwdreqp, sizeof *rwdreqp);

        /* Copy cchhr */

        copyin((caddr_t)rwdreqp->cchhr, (caddr_t)cchhr, 5);

	/* Copy key, if it's there. */

        i = 0;
        if (rwdreqp->key){
                p1 = key;
                p2 = rwdreqp->key;
                for (;i<255;i++)
                        if (!(*p1++ = fubyte(p2++)))
                                break;
	}
        return(i);
}
