/*
 * ASCII terminal driver.
 *
 * The relevant IBM manual is GA22-6864,
 * "Component Description: IBM 2701 Data Adapter Unit".
 * See the section on start/stop transmission adapters.
 */
#include "../h/param.h"
#include "../h/systm.h"
#include "../h/conf.h"
#include "../h/signal.h"
#include "../h/tty.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/io.h"
#include "../h/ioconf.h"

#define min(a,b)        ((a < b) ? a : b)

/* Input an Output buffer sizes - must both fit into a page(PSIZE) */
#define ISIZE   2048
#define OSIZE   2048
 
/* CCWs */
#define WRITE   0x01
#define PREPARE 0x06
#define INHIBIT 0x0a
#define ENABLE  0x27
#define DISABLE 0x2f
 
/* sense bits */
#define INTREQ  0x40
#define DATACHK 0x08
#define TIMEOUT 0x01
#define BREAK   (INTREQ|DATACHK)
 
#define MAXERRS 5              /* max UCs before hangup */
 
/* States that the terminal driver is in when expecting an interrupt */
enum tstates {
	INACTIVE = 0,
	ENABLING,
	WAITING,
	READING,
	HALTING,
	WRITING,
	PREPARING,
	DISABLING,
};
 
struct asciiterm {     /* ascii terminal */
        ccw_t   t_ccws[2];      /* ccws */
        caddr_t t_bp;           /* input and output buffer */
	enum tstates t_st;      /* state */
        int     t_il;           /* input length */
        int     t_ol;           /* output length */
        char    *t_base;        /* output addr */
        int     t_wresid;       /* output residual */
        char    t_lf;           /* immed line feed */
        char    t_enb;          /* line enabled */
        int     t_errcnt;       /* unit check count */
} terms[NTRMS];
 
struct tty trms[NTRMS];
 
extern char revbit[];
int trmintr();
 
/*
 * Upper level routines: trmopen, trmclose, trmread, trmwrite,
 * trmsgtty, and trmstart.
 */
 
 
/*
 * Open a terminal.  If already open, enable line if not already enabled.  If
 * not open, set default values and enable line.
 */
/* ARGSUSED */
trmopen(dev,flag)
{
        struct tty *tp;
        struct asciiterm *atp;
        int trmstart();

        if(minor(dev) >= NTRMS){
                u.u_error = ENXIO;
                return;
        }
        tp = &trms[minor(dev)];
        atp = &terms[minor(dev)];
        if(tp->t_state & ISOPEN) {    /* already open */
		if(!atp->t_enb)
			trmenbl(tp);
		ttyopen(dev, tp);
		return;
	}
        atp->t_errcnt = 0;
        tp->t_dev = dev;
        tp->t_addr = cdevsw[major(dev)].d_addrs[minor(dev)];
        tp->t_oproc = trmstart;
        tp->t_flags = XTABS|CRMOD;
        tp->t_ispeed = tp->t_ospeed = 1;       /* non-zero */
	ttychars(tp);
        tp->t_erase = CERASE;
        tp->t_kill = CKILL;
        /* Get input and output buffers */
        atp->t_bp = getpage();
	tp->t_state = WOPEN;
        if(!atp->t_enb) {
		trmenbl(tp);
        } else { /* Start Reading */
		tp->t_state |= CARR_ON;
                trmnorm(tp);
        }
	ttyopen(dev, tp);
}
 
/*
 * Close a terminal. Flush queues, halt activity, and disable line.
 */
trmclose(dev)
{
        struct tty *tp;
        struct asciiterm *atp;
	enum tstates st;

        tp = &trms[minor(dev)];
        atp = &terms[minor(dev)];
        if(atp->t_st == INACTIVE || atp->t_st == DISABLING)
                flushtty(tp);
        else {
                wflushtty(tp);
                if(tp->t_state & HUPCLS)
                        trmhup(tp, DISABLING);
                else {
			st = atp->t_st;
                        atp->t_st = DISABLING;
                        if(st == WAITING)
                                hio(tp->t_addr);
                }
        }
        while(atp->t_st != INACTIVE) sleep((caddr_t)&atp->t_st, TTOPRI);
        freepag(atp->t_bp);
	ttyclose(tp);
}
 
trmread(dev)
{
        ttread(&trms[minor(dev)]);
}
 
trmwrite(dev)
{
        ttwrite(&trms[minor(dev)]);
}
 
/* ARGSUSED */
trmioctl(dev, cmd, arg, flag)
dev_t dev;
caddr_t arg;
{
        struct tty *tp;
 
        tp = &trms[minor(dev)];
	if(ttioccomm(cmd, tp, arg, dev)) {
		if(tp->t_ospeed == 0) {
                        wflushtty(tp);
                        trmhup(tp, DISABLING);
                        return;
                }
	} else {
		u.u_error = ENOTTY;
                return;
	}
}
 
/*
 * Start output to a terminal.
 * If we waiting for something to be typed,
 * halt read and start writing.
 */
trmstart(tp)
struct tty *tp;
{
	struct asciiterm *atp;

	atp = &terms[minor(tp->t_dev)];
        if(atp->t_st == WAITING){
                atp->t_st = HALTING;
                hio(tp->t_addr);
        }
}
 
/*
 * Terminal interrupt routine. Entire routine is run disabled.
 */
trmintr(tp, csw, sense)
struct tty *tp;
csw_t *csw;
char *sense;
{
        struct asciiterm *atp;
        int len;
        char *cp;

        if(tp == 0) return; /* hangup interrupt */
        atp = &terms[minor(tp->t_dev)];
 
        /* Huge switch by current state */
        switch(atp->t_st){
 
        case INACTIVE: /* nothing happening */
                return;
 
        case ENABLING:
                /* someone has dialed up */
                if(csw->cs_de) {
			tp->t_state |= CARR_ON;
			atp->t_enb = 1;
			wakeup((caddr_t)&atp->t_enb);
                        trmnorm(tp);
		}
                return;
 
        case WAITING:
        case READING:
                if(csw->cs_pci && (csw->cs_word2 & UNITSTAT) == 0) {
                        /* typing begun, but not finished */
                        atp->t_st = READING;
                        return;
                }
                if(csw->cs_il) {
                        /* break hit with no chars typed */
                        trmprep(tp);
                        return;
                }
                /* terminal input */
                len = atp->t_il - csw->cs_count;
                cp = atp->t_bp;
                while(len--) ttyinput(revbit[*cp++],tp); /* process chars */
                if(csw->cs_uc && (*sense & BREAK)) {
                        trmprep(tp);
                        return;
                }
                /* normally terminated read */
                switch(revbit[*--cp] & 0177) {
                case CINTR:
                case CQUIT:
                        /* don't restart write */
                        atp->t_wresid = 0;
                        break;
                case '\r':
                        /* immediate lf */
                        atp->t_lf = 1;
                }
                trmnorm(tp);
                return;
 
        case HALTING:
                /* read was halted to begin write */
                trmnorm(tp);
                return;
 
        case WRITING:
                /* data written */
                atp->t_wresid = csw->cs_count;
                if(csw->cs_uc && (*sense & BREAK)) {
                        /* write interrupted by break */
                        trmprep(tp);
                        return;
                }
                trmnorm(tp);
                return;
 
        case PREPARING:
                /* Waiting for line to come to life. */
                if(csw->cs_uc && (*sense & (BREAK|TIMEOUT))) {
                        if((*sense & TIMEOUT) || ++atp->t_errcnt >= MAXERRS)
                                trmhup(tp, INACTIVE);
                        else
                                trmprep(tp);
                        return;
                }
                trmrd(tp);
                atp->t_st = READING;
                return;
 
        case DISABLING:
		tp->t_state &= ~CARR_ON;
                atp->t_st = INACTIVE;
                wakeup((caddr_t)&atp->t_st);
                return;
 
        }       /* end of state switch */
}
 
/*
 * Perform normal action for the terminal.
 * If we need to restart an interrupted write, do so,
 * if there is anything to write, write it,
 * else read.
 */
trmnorm(tp)
struct tty *tp;
{
        int i,len;
        char *cp;
        struct asciiterm *atp;
 
        atp = &terms[minor(tp->t_dev)];
        if(atp->t_wresid > 0) {  /* restart write */
                atp->t_base += (atp->t_ol - atp->t_wresid);
                atp->t_ol = atp->t_wresid;
                if(atp->t_lf) {
                        *--(atp->t_base) = revbit['\n'];
                        atp->t_ol++;
                        atp->t_lf = 0;
                }
                atp->t_st = WRITING;
		atp->t_ccws[0].cc_dblw = 0;
                atp->t_ccws[0].cc_cmd = WRITE;
                atp->t_ccws[0].cc_addr = (int)atp->t_base;
                atp->t_ccws[0].cc_sli = 1;
                atp->t_ccws[0].cc_count = atp->t_ol;
                sio(tp->t_addr, atp->t_ccws, trmintr, (int)tp);
                return;
        }
        if((!atp->t_lf) && tp->t_outq.c_cc == 0){       /* read */
                trmrd(tp);
                return;
        }
        /* write */
        cp = atp->t_base = atp->t_bp + ISIZE;
        if(atp->t_lf) *cp++ = revbit['\n'];
        len = min(OSIZE, tp->t_outq.c_cc);
        for(i=0;i<len;i++)
                *cp++ = revbit[getc(&tp->t_outq)];
        atp->t_st = WRITING;
        atp->t_ol = len + atp->t_lf;
        atp->t_lf = 0;
	atp->t_ccws[0].cc_dblw = 0;
        atp->t_ccws[0].cc_cmd = WRITE;
        atp->t_ccws[0].cc_addr = (int)atp->t_base;
        atp->t_ccws[0].cc_sli = 1;
        atp->t_ccws[0].cc_count = atp->t_ol;
        sio(tp->t_addr, atp->t_ccws, trmintr, (int)tp);
	if(tp->t_state & ASLEEP)
		wakeup((caddr_t)&tp->t_outq);
}
 
/*
 * Read!
 */
trmrd(tp)
struct tty *tp;
{
        struct asciiterm *atp;
 
        atp = &terms[minor(tp->t_dev)];
        atp->t_st = WAITING;
        atp->t_il = ISIZE;
	atp->t_ccws[0].cc_dblw = 0;
        atp->t_ccws[0].cc_cmd = INHIBIT;
        atp->t_ccws[0].cc_addr = (int)atp->t_bp;
        atp->t_ccws[0].cc_cd = 1;
        atp->t_ccws[0].cc_sli = 1;
        atp->t_ccws[0].cc_count = 1;
	atp->t_ccws[1].cc_dblw = 0;
        atp->t_ccws[1].cc_cmd = INHIBIT;
        atp->t_ccws[1].cc_addr = (int)(atp->t_bp + 1);
        atp->t_ccws[1].cc_sli = 1;
        atp->t_ccws[1].cc_pci = 1;
        atp->t_ccws[1].cc_count = atp->t_il - 1;
        sio(tp->t_addr, atp->t_ccws, trmintr, (int)tp);
}
 
/*
 * Issue prepare.
 * Done after BREAK to check for hung up line, or normal break.
 */
trmprep(tp)
struct tty *tp;
{
        struct asciiterm *atp;
 
        atp = &terms[minor(tp->t_dev)];
        atp->t_st = PREPARING;
	atp->t_ccws[0].cc_dblw = 0;
        atp->t_ccws[0].cc_cmd = PREPARE;
        atp->t_ccws[0].cc_sli = 1;
        atp->t_ccws[0].cc_count = 1;
        sio(tp->t_addr, atp->t_ccws, trmintr, (int)tp);
        /* Now we halt it to start timeout */
        hio(tp->t_addr);
        /* see manual */
}
 
/*
 * Hang up processing.
 * Send signal, disable line.
 */
trmhup(tp, nstate)
struct tty *tp;
enum tstates nstate;
{
        struct asciiterm *atp;
	enum tstates st;

        atp = &terms[minor(tp->t_dev)];
	st = atp->t_st;
	atp->t_st = nstate;
        if(st == WAITING)
                hio(tp->t_addr);
        if(atp->t_st == INACTIVE) wakeup((caddr_t)&atp->t_st);
        atp->t_errcnt = 0;
        atp->t_enb = 0;
	atp->t_ccws[0].cc_dblw = 0;
        atp->t_ccws[0].cc_cmd = DISABLE;
        atp->t_ccws[0].cc_sli = 1;
        atp->t_ccws[0].cc_count = 1;
        sio(tp->t_addr, atp->t_ccws, trmintr, 0);
        signal(tp->t_pgrp, SIGHUP);
	tp->t_state &= ~CARR_ON;
}

/*
 * Enable the terminal
 */
trmenbl(tp)
struct tty *tp;
{
	struct asciiterm *atp;

	atp = &terms[minor(tp->t_dev)];
        atp->t_st = ENABLING;
	atp->t_ccws[0].cc_dblw = 0;
        atp->t_ccws[0].cc_cmd = ENABLE;
        atp->t_ccws[0].cc_sli = 1;
        atp->t_ccws[0].cc_count = 1;
        sio(tp->t_addr, atp->t_ccws, trmintr, (int)tp);
	while(!atp->t_enb) sleep((caddr_t)&atp->t_enb, TTIPRI);
}
