/*
 * 3270 terminal driver.
 *
 * Actually this is two drivers in one:
 * 1. A "normal mode" driver, in which the 3270 seems to be
 *    a normal ascii terminal (nearly full duplex).
 *    The routines of this driver start with the prefix "tub".
 * 2. A "full-screen mode" driver in which the 3270 is available
 *    to the user process with all it's features and deficiencies.
 *    The routines of this driver start with the prefix "fs".
 *
 * This driver handles local 3277 terminals attached to a 3272 controller.
 * The IBM manual which describes the 3270 is GA27-2749
 */

#include "../h/param.h"
#include "../h/conf.h"
#include "../h/signal.h"
#include "../h/proc.h"
#include "../h/systm.h"
#include "../h/tty.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/io.h"
#include "../h/ioconf.h"
#include "../h/tube.h"

/* Priorities to sleep at while waiting for wakeup */
/* Must be kept with same relative values */
#define RPRI    (PZERO+1)         /* read */
#define WPRI    (PZERO+2)         /* write */
#define OPRI    (PZERO+3)         /* open */
/* Full screen should be less than PZERO so the sleep can't be interrupted */
/* Otherwise the process could exit while i/o was in progress to it's address space */
#define FSRPRI  (PZERO-2)         /* read */
#define FSWPRI  (PZERO-1)         /* write */

struct tty tubes[NTUBES];
static struct tube *tptrs[NTUBES];

char moremsg[] = "More...";
char ucmsg[] = "Unit check on terminal, data lost.";

extern char tubecc[], tubitab[], tubotab[], mrxotab[];
int tubstart(), tuboaint(), tubowint(), tubaint(), tubrwint();
char *tubochars(), *tubinput(), *sba(), *ra(), *bufinit();

/*
 * Open a tube for normal use.
 * ARGSUSED
 */
tubopen(dev, flag)
register dev_t dev;
int flag;
{
	register struct tty *tp;
	register struct tube *ttp;
	register char *dbufp;
	register ccw_t *ccwp;

	if (minor(dev) >= NTUBES) {
		u.u_error = ENXIO;
		return;
	}
	tp = &tubes[minor(dev)];
	ttp = tptrs[minor(dev)];
	if (ttp == 0) {                   /* not allocated */
		ttp = tptrs[minor(dev)] = (struct tube *) getpage();
		tp->t_addr = cdevsw[major(dev)].d_addrs[minor(dev)];
		tp->t_dev = dev;
		tp->t_oproc = tubstart;
		tp->t_flags = ECHO|CRMOD|IDBL|ODBL;
		tp->t_ispeed = tp->t_ospeed = 1;       /* non-zero */
		ttychars(tp);
		tp->t_erase = tp->t_kill = 0;
		tp->t_state = WOPEN;
		ttp->t_re_back = 10;
		ttp->t_re_forw = 11;
		ttp->t_re_off = NRE;
		ttp->t_pfk[0][0] = CEOT;
		ccwp = ttp->t_ccws;
		ccwp->cc_dblw = 0;
		ccwp->cc_cmd = SELECT;
		ccwp->cc_cc = 1;
		ccwp->cc_sli = 1;
		ccwp->cc_count = 1;
		ccwp++;
		ccwp->cc_dblw = 0;
		ccwp->cc_addr = (int) ttp->t_dbuf;
		ccwp->cc_sli = 1;
	}
	if ((tp->t_state & CARR_ON) == 0) {
                setax(tp->t_addr, tuboaint, (int) tp);
                while ((tp->t_state & CARR_ON) == 0) {
                        if ((tp->t_state & BUSY) == 0) {
                                tp->t_state |= BUSY;
                                dbufp = bufinit(tp, RESTORE);
                                *dbufp++ = IC;
                                ttp->t_ccws[1].cc_cmd = EWRITE;
                                ttp->t_ccws[1].cc_count = dbufp - ttp->t_dbuf;
                                sio(tp->t_addr, ttp->t_ccws + 1, tubowint, (int) tp);
                        }
                        while (tp->t_state & BUSY) {
                                ttp->t_bits |= T_ASLEEP;
                                sleep((caddr_t) ttp, OPRI);
                        }
                }
	}
        ttyopen(dev, tp);
}

/*
 * Open a tube for full screen use.
 * ARGSUSED
 */
fsopen(dev, flag)
register dev_t dev;
int flag;
{
	register struct tty *tp;
	register struct tube *ttp;

	if (minor(dev) >= NTUBES) {                     /* invalid device number */
		u.u_error = ENXIO;
		return;
	}
	tp = &tubes[minor(dev)];
	ttp = tptrs[minor(dev)];
	if (ttp == 0 || (tp->t_state & ISOPEN) == 0) {  /* normal tube not open */
		u.u_error = ENXIO;
		return;
	}
	if (ttp->t_bits & (T_FS | T_FSNORM)) {          /* full screen already in use */
		u.u_error = EBUSY;
		return;
	}
	if (tp != u.u_ttyp && !suser())                 /* not controlling tty */
		return;
	/* enough checking - now do the open */
	wflushtty(tp);
	while (tp->t_state & BUSY) {  /* wait for outstanding i/o to complete */
		ttp->t_bits |= T_ASLEEP;
		sleep((caddr_t) ttp, WPRI);
	}
	ttp->t_bits |= T_FS;
	ttp->t_ccws[1].cc_addr = (int) ttp->t_idaws;
	ttp->t_ccws[1].cc_ida = 1;
	ttp->t_bits &= ~(T_NOAT + T_ATTN);
	ttp->t_rcmd = READMOD;
	ttp->t_wcmd = EWRITE;
	ttp->t_pksig = 0;
}

/*
 * Close a tube.
 * It would be logcal to free the tube structure in this routine,
 * but we can't be sure the tube isn't still in full screen mode.
 * We could set a flag and have fsclose try to do it, but there
 * are conditions under which this doesn't work either.
 * Besides, /etc/init is going to come along very quickly and
 * open the tube again anyway, so the page would only be free
 * for a short time.
 */
tubclose(dev)
register dev_t dev;
{
	register struct tty *tp;
	register struct tube *ttp;
	register int hup;

	tp = &tubes[minor(dev)];
	ttp = tptrs[minor(dev)];
	hup = tp->t_state & HUPCLS;
	ttyclose(tp);
	ttp->t_curpos = 0;
	if (hup) {
		reset (tp->t_addr);
		signal(tp->t_pgrp, SIGHUP);
	}
}


/*
 * Close full screen mode.
 */
fsclose(dev)
register dev_t dev;
{
	register struct tty *tp;
	register struct tube *ttp;
	register char *dbufp;

	tp = &tubes[minor(dev)];
	ttp = tptrs[minor(dev)];
	ttp->t_bits &= ~T_FS;
	ttp->t_ccws[1].cc_addr = (int) ttp->t_dbuf;
	ttp->t_ccws[1].cc_ida = 0;
	if (tp->t_state && (tp->t_state & CARR_ON))
                if (ttp->t_bits & T_FSNORM)  /* normal interruption in progress */
                        ttp->t_bits &= ~T_FSNORM;
                else {
                        while (tp->t_state & BUSY) {
                                ttp->t_bits |= T_ASLEEP;
                                sleep((caddr_t) ttp, WPRI);
                        }
                        ttp->t_curpos = 0;
                        tp->t_state |= BUSY;
                        dbufp = bufinit(tp, RESTORE);
                        *dbufp++ = IC;
                        dbufp = tubochars(tp, dbufp);
                        ttp->t_ccws[1].cc_cmd = EWRITE;
                        ttp->t_ccws[1].cc_count = dbufp - ttp->t_dbuf;
                        sio(tp->t_addr, ttp->t_ccws + 1, tubrwint, (int) tp);
                }
}

/*
 * Read from a tube.
 */
tubread(dev)
register dev_t dev;
{
	ttread(&tubes[minor(dev)]);
}

/*
 * Full screen read.
 */
fsread(dev)
register dev_t dev;
{
	register struct tty *tp;
	register struct tube *ttp;
	register int recheck;
	register char c;

	tp = &tubes[minor(dev)];
	ttp = tptrs[minor(dev)];
	if ((tp->t_state & CARR_ON) == 0) {
		u.u_error = EIO;
		return;
	}
	/*
	 * The following loop ensures that:
	 *   1. No normal interruption is in progress,
	 *   2. We have the attention if we need it,
	 *   3. The tube is not busy.
	 * With these assured we can do the read.
	 */
	do {
		recheck = 0;
		if (ttp->t_bits & T_FSNORM) {
			sleep((caddr_t) &ttp->t_bits, RPRI);
			recheck = 1;
		}
		if ((ttp->t_bits & (T_NOAT | T_ATTN)) == 0) {
			sleep((caddr_t) &ttp->t_bits, RPRI);
			recheck = 1;
		}
		if (tp->t_state & BUSY) {
			ttp->t_bits |= T_ASLEEP;
			sleep((caddr_t) ttp, RPRI);
			recheck = 1;
		}
	} while (recheck);
	makidaw(0, ttp->t_idaws, u.u_base, u.u_count);
	if (u.u_error)
		return;
	tp->t_state |= BUSY;
	ttp->t_ccws[1].cc_cmd = ttp->t_rcmd;
	ttp->t_ccws[1].cc_count = (int) u.u_count;
	sio(tp->t_addr, ttp->t_ccws, tubrwint, (int) tp);
	do {
		ttp->t_bits |= T_ASLEEP;
		sleep((caddr_t) ttp, FSRPRI);
	} while (tp->t_state & BUSY);
	ttp->t_bits &= ~T_ATTN;
	if (ttp->t_resid == -1)
		u.u_error = EIO;
	else
		u.u_count = ttp->t_resid;
	c = * (char *) ttp->t_idaws[0];
	if (c == PA1)
		signal(tp->t_pgrp, SIGINTR);
	if (c == CLEAR && tp->t_outq.c_cc > 0)
		fsnorm(tp);
}

/*
 * Write to a tube.
 */
tubwrite(dev)
register dev_t dev;
{
	ttwrite(&tubes[minor(dev)]);
}

/*
 * Full screen write.
 */
fswrite(dev)
register dev_t dev;
{
	register struct tty *tp;
	register struct tube *ttp;
	register int recheck;

	tp = &tubes[minor(dev)];
	ttp = tptrs[minor(dev)];
	if ((tp->t_state & CARR_ON) == 0) {
		u.u_error = EIO;
		return;
	}
	while (ttp->t_bits & T_FSNORM)
		sleep((caddr_t) &ttp->t_bits, WPRI);
	if (ttp->t_wcmd == EWRITE && tp->t_outq.c_cc > 0)
		fsnorm(tp);
	/*
	 * The following loop ensures that:
	 *   1. No normal interruption is in progress,
	 *   2. The tube is not busy.
	 */
	do {
		recheck = 0;
		if (ttp->t_bits & T_FSNORM) {
			sleep((caddr_t) &ttp->t_bits, WPRI);
			recheck = 1;
		}
		if (tp->t_state & BUSY) {
			ttp->t_bits |= T_ASLEEP;
			sleep((caddr_t) ttp, WPRI);
			recheck = 1;
		}
	} while (recheck);
	makidaw(1, ttp->t_idaws, u.u_base, u.u_count);
	if (u.u_error)
		return;
	tp->t_state |= BUSY;
	ttp->t_ccws[1].cc_cmd = ttp->t_wcmd;
	ttp->t_ccws[1].cc_count = (int) u.u_count;
	if (ttp->t_wcmd == EWRITE)     /* manual says to skip select on erase write */
		sio(tp->t_addr, ttp->t_ccws + 1, tubrwint, (int) tp);
	else
		sio(tp->t_addr, ttp->t_ccws, tubrwint, (int) tp);
	do {
		ttp->t_bits |= T_ASLEEP;
		sleep((caddr_t) ttp, FSWPRI);
	} while (tp->t_state & BUSY);
	if (ttp->t_resid == -1)
		u.u_error = EIO;
	else
		u.u_count = ttp->t_resid;
}

/*
 * Ioctl for tubes.
 * ARGSUSED
 */
tubioctl(dev, cmd, argp, flag)
dev_t dev;
caddr_t argp;
{
	struct tty *tp;
	register struct tube *ttp;
	register char c;
	register char *cp;
	register caddr_t up;
	register int i;
	int n, oflags;

	tp = &tubes[minor(dev)];
	ttp = tptrs[minor(dev)];
	if (tp != u.u_ttyp && !suser())
		return;
	oflags = tp->t_flags;
	if (ttioccomm(cmd, tp, argp, dev)) {
		if (ttp->t_bits & T_FS)
			return;
		if (tp->t_ospeed == 0) {       /* force hangup */
			wflushtty(tp);
			reset(tp->t_addr);
			tp->t_state &= ~(CARR_ON | BUSY);
			signal(tp->t_pgrp, SIGHUP);
			return;
		}
		oflags = oflags ^ tp->t_flags;  /* get changed bits */
		if (oflags & ECHO) {
			/* change displayability of input field */
			if (tp->t_outq.c_cc == 0)
				putc('\0', &tp->t_outq); /* trick tubstart so it will reformat screen */
			tubstart(tp);
		}
		return;
	}

	switch(cmd) {

	case TUBSETPF:                /* set PF key */
		n = fuword(argp);
		up = (caddr_t)fuword(argp+4);
		if (n < 0 || n > NPFK) {
			u.u_error = EINVAL;
			return;
		}
		switch (up) {

		case 0:
			ttp->t_re_back = n;
			*ttp->t_pfk[n] = '\0';
			break;

		case -1:
			ttp->t_re_forw = n;
			*ttp->t_pfk[n] = '\0';
			break;

		default:
			cp = ttp->t_pfk[n];
			for (i=0; i<PFLEN; i++) {
				if ((c = fubyte(up++)) <= 0)
					break;
				*cp++ = c;
			}
			if (n == ttp->t_re_back)
				ttp->t_re_back = -1;
			else if (n == ttp->t_re_forw)
				ttp->t_re_forw = -1;
			*cp = '\0';
		}
		break;

	case TUBGETPF:          /* get PF key */
		n = fuword(argp);
		up = (caddr_t)fuword(argp+4);
		if (n < 0 || n > NPFK) {
			u.u_error = EINVAL;
			return;
		}
		if (n == ttp->t_re_back)
			subyte(up++, 0376);
		else if (n == ttp->t_re_forw)
			subyte(up++, 0375);
		else {
			cp = ttp->t_pfk[n];
			for (i=0; i<PFLEN; i++) {
				if (*cp == 0)
					break;
				subyte(up++, *cp++);
			}
		}
		subyte(up++, '\0');
		break;

	default:
		u.u_error = ENOTTY;
		break;
	}
}

/*
 * Full screen ioctl.
 * ARGSUSED
 */
fsioctl(dev, cmd, argp, flag)
register dev_t dev;
register int cmd;
register caddr_t argp;
int flag;
{
	register struct tube *ttp;
	register int n;

	ttp = tptrs[minor(dev)];

	switch(cmd) {

	case TUBICMD:   /* set fs read ccw */
		n = (int)argp;
		if(n != READ && n != READMOD) {
			u.u_error = EINVAL;
			return;
		}
		ttp->t_rcmd = n;
		break;

	case TUBOCMD:   /* set fs write ccw */
		n = (int)argp;
		if(n != WRITE && n != EWRITE && n != WEUA) {
			u.u_error = EINVAL;
			return;
		}
		ttp->t_wcmd = n;
		break;

	case TUBWATTN:
		ttp->t_bits &= ~T_NOAT;
		break;

	case TUBNATTN:
		ttp->t_bits |= T_NOAT;
		break;

	case TUBGETI:
		if(subyte(argp, ttp->t_rcmd))
			u.u_error = EFAULT;
		break;

	case TUBGETO:
		if(subyte(argp, ttp->t_wcmd))
			u.u_error = EFAULT;
		break;

	case TIOPOLL:
		if(suword(argp, (ttp->t_bits & T_ATTN) ? 1 : 0))
			u.u_error = EFAULT;
		break;

	case TIOPOKE:
		ttp->t_pksig = (int) argp;
		ttp->t_pkproc = u.u_procp;
		break;

	case TIONPOKE:
		ttp->t_pksig = 0;
		break;

	default:
		u.u_error = ENOTTY;
		break;
	}
}

/*
 * Normal io during full screen mode.
 * In full screen mode
 *    if ((the user hits clear || the program does an erase write)
 *                && (there are characters on the normal output queue))
 *    then we allow a "normal interruption in full screen mode"
 *                in which we ring the bell and output the normal
 *                output data to the terminal, returning to full
 *                screen mode when the user hits the clear key
 *                and the normal mode output is exhausted.
 * fsnorm is called to start this process, fsrestor to end it.
 */
fsnorm(tp)
struct tty *tp;
{
	register struct tube *ttp;
	register char *dbufp;

	ttp = tptrs[minor(tp->t_dev)];
	while (tp->t_state & BUSY) {
		ttp->t_bits |= T_ASLEEP;
		sleep((caddr_t) ttp, WPRI);
	}
	ttp->t_bits |= T_FSNORM;
	ttp->t_bits &= ~T_FS;
	ttp->t_ccws[1].cc_addr = (int) ttp->t_dbuf;
	ttp->t_ccws[1].cc_ida = 0;
	ttp->t_curpos = 0;
	tp->t_state |= BUSY;
	dbufp = bufinit(tp, RESTORE | ALARM);
	*dbufp++ = IC;
	dbufp = tubochars(tp, dbufp);
	ttp->t_ccws[1].cc_cmd = EWRITE;
	ttp->t_ccws[1].cc_count = dbufp - ttp->t_dbuf;
	sio(tp->t_addr, ttp->t_ccws + 1, tubrwint, (int) tp);
}

/*
 * Restore full screen mode after interruption for normal output.
 */
fsrestor(tp)
register struct tty *tp;
{
	register struct tube *ttp;

	ttp = tptrs[minor(tp->t_dev)];
	flushtty(tp);
	ttp->t_bits &= ~T_FSNORM;
	ttp->t_bits |= T_FS;
	ttp->t_ccws[1].cc_addr = (int) ttp->t_idaws;
	ttp->t_ccws[1].cc_ida = 1;
}

/*
 * Start output to a tube.
 * Called at both process and interrupt levels by ttyoutput.
 * Also called by tubioctl.
 */
tubstart(tp)
register struct tty *tp;
{
	register struct tube *ttp;
	register char *dbufp;

	ttp = tptrs[minor(tp->t_dev)];
	if (tp->t_state & BUSY)                                  /* will be called soon by tubrwint */
		return;
	if (ttp->t_curpos > MAXPOS && ttp->t_bits & T_MOREMSG)   /* screen full */
		return;
	if (ttp->t_bits & T_FS)                                  /* full screen in use */
		return;
	tp->t_state |= BUSY;
	dbufp = bufinit(tp, 0);
	dbufp = tubochars(tp, dbufp);
	ttp->t_ccws[1].cc_cmd = WRITE;
	ttp->t_ccws[1].cc_count = dbufp - ttp->t_dbuf;
	sio(tp->t_addr, ttp->t_ccws, tubrwint, (int) tp);
}

/*
 * Output normal mode data (i.e., move chars
 * from output queue to device buffer).
 */
char *
tubochars(tp, dbufp)
register struct tty *tp;
register char *dbufp;
{
	register struct tube *ttp;
	register int physpos;
	register char c;
	char *cp, ring;

	ttp = tptrs[minor(tp->t_dev)];
	physpos = -1;
	ring = 0;
	while (tp->t_outq.c_cc > 0 && ttp->t_curpos <= MAXPOS && dbufp < &ttp->t_dbuf[BUFLEN-FUDGE]) {
		switch(c = getc(&tp->t_outq)) {

		case CR:
			ttp->t_curpos -= ttp->t_curpos % 80;
			break;

		case TAB:
			ttp->t_curpos = (ttp->t_curpos+8) & ~07;
			break;

		case FF:
			dbufp = sba(dbufp, 0);
			dbufp = ra(dbufp, MAXPOS, 0);
			physpos = MAXPOS;
			ttp->t_curpos = 0;
			break;

		case BELL:
			ring = 1;
			break;

		case LF:
			if (ttp->t_bits & T_SKLF)
				ttp->t_bits &= ~T_SKLF;
			else
				ttp->t_curpos += 80;
			break;

		case BS:
			if (ttp->t_curpos > 0)
				ttp->t_curpos--;
			if (ttp->t_curpos % 80 == 79)
				ttp->t_bits &= ~T_SKLF;
			break;

		case '\0':
			break;

		default:
			if (physpos != ttp->t_curpos)
				dbufp = sba(dbufp, physpos = ttp->t_curpos);
			if (tp->t_flags & ODBL)
				*dbufp++ = tubotab[c & 0177];
			else
				*dbufp++ = mrxotab[c];
			ttp->t_curpos++;
			physpos++;
			if (ttp->t_curpos % 80 == 0)
				ttp->t_bits |= T_SKLF;
			else
				ttp->t_bits &= ~T_SKLF;
			break;
		}
	}
	if (ttp->t_curpos > MAXPOS && tp->t_outq.c_cc) { /* write more msg */
		if (ttp->t_curpos == MAXPOS+1)
			ttp->t_bits |= T_SKLF;
		dbufp = sba(dbufp, MSGPOS);
		cp = moremsg;
		while (*cp)
			*dbufp++ = tubotab[*cp++];
		ttp->t_bits |= T_MOREMSG;
	}
	if (ring) {
		c = ttp->t_dbuf[0];          /* pick up orig wcc */
		c &= 077;                    /* remove high bits */
		c |= ALARM;                  /* or in alarm bit */
		ttp->t_dbuf[0] = tubecc[c];  /* get appropriate high bits */
	}
	if (tp->t_state & ASLEEP) {
		wakeup((caddr_t) &tp->t_outq.c_cc);
		tp->t_state &= ~ASLEEP;
	}
	return(dbufp);
}

/*
 * Open in progress asynchronous interrupt.
 * ARGSUSED
 */
tuboaint(tp, csw, sense)
register struct tty *tp;
csw_t *csw;
char *sense;
{
	register struct tube *ttp;

	ttp = tptrs[minor(tp->t_dev)];
	if (csw->cs_de)                  /* device ready */
		sio(tp->t_addr, ttp->t_ccws, tubowint, (int) tp);
}

/*
 * Open in progress write interrupt.
 * ARGSUSED
 */
tubowint(tp, csw, sense)
register struct tty *tp;
register csw_t *csw;
char *sense;
{
	register struct tube *ttp;

	ttp = tptrs[minor(tp->t_dev)];
	if (csw->cs_cc == 3 || csw->cs_uc)   /* wait for asychronous DE to reissue sio */
		return;
	if (csw->cs_ue && csw->cs_de) {
		sio(tp->t_addr, ttp->t_ccws, tubowint, (int) tp);
		return;
	}
	if (csw->cs_de) {
		tp->t_state |= CARR_ON;
		tp->t_state &= ~BUSY;
		setax(tp->t_addr, tubaint, (int) tp);
		wakeup((caddr_t) ttp);
		ttp->t_bits &= ~T_ASLEEP;
	}
}

/*
 * Asynchronous interrupt.
 * ARGSUSED
 */
tubaint(tp, csw, sense)
register struct tty *tp;
register csw_t *csw;
char *sense;
{
	register struct tube *ttp;

	ttp = tptrs[minor(tp->t_dev)];
	tp->t_state |= CARR_ON;
	if (csw->cs_de)
		if (ttp->t_bits & T_PEND) {
			if (ttp->t_ccws[1].cc_cmd == EWRITE)
			        sio(tp->t_addr, ttp->t_ccws + 1, tubrwint, (int) tp);
			else
			        sio(tp->t_addr, ttp->t_ccws, tubrwint, (int) tp);
			ttp->t_bits &= ~T_PEND;
		} else {        /* due to pa1 feature of session */
			signal(tp->t_pgrp, SIGHUP);
			flushtty(tp);
		}
	else if (csw->cs_attn)
		if ((ttp->t_bits & T_FS) == 0) {   /* normal mode */
			if ((tp->t_state & BUSY) == 0) {  /* start read */
				tp->t_state |= BUSY;
				ttp->t_ccws[1].cc_cmd = READMOD;
				ttp->t_ccws[1].cc_count = BUFLEN;
				sio(tp->t_addr, ttp->t_ccws, tubrwint, (int) tp);
				ttp->t_bits |= T_RIP;
			} else              /* tubrwint will handle it */
				ttp->t_bits |= T_ATTN;
		} else {                            /* full screen mode */
			if (ttp->t_pksig > 0) {
				psignal(ttp->t_pkproc, ttp->t_pksig);
				ttp->t_pksig = 0;
			}
			ttp->t_bits |= T_ATTN;
			wakeup((caddr_t) &ttp->t_bits);
		}
}

/*
 * Read/write interrupt.
 */
tubrwint(tp, csw, sense)
register struct tty *tp;
register csw_t *csw;
char *sense;
{
	register struct tube *ttp;
	register char *dbufp, *cp;

	ttp = tptrs[minor(tp->t_dev)];
	if (csw->cs_cc == 3) {
		printf("CC 3 on tube %x\n", tp->t_dev);
		tp->t_state &= ~(CARR_ON | BUSY);
		ttp->t_resid = -1;
	} else if (csw->cs_ue) {
		printf("Unit exception on tube %x csw %x %x\n", tp->t_addr, csw->cs_dblw);
		if (csw->cs_de)
			if (ttp->t_ccws[1].cc_cmd == EWRITE)
			        sio(tp->t_addr, ttp->t_ccws + 1, tubrwint, (int) tp);
			else
			        sio(tp->t_addr, ttp->t_ccws, tubrwint, (int) tp);
		else
			ttp->t_bits |= T_PEND;
	} else if (csw->cs_uc) {
		printf("Unit check on tube %x csw %x %x sense %x\n", tp->t_addr, csw->cs_dblw, *sense);
		ttp->t_errct++;
		if (*sense & (EQCHK | UNITSPC) && ttp->t_errct < 3)
			ttp->t_errct = 3;
		if (*sense & (COMREJ | OPCHK | INTREQ))
			ttp->t_errct = 5;
		switch (ttp->t_errct) {
		case 1:
		case 2:
		case 4:
			sio(tp->t_addr, ttp->t_ccws, tubrwint, (int) tp);
			break;
		case 3:
			/* manual says to rebuild screen, but we don't have it */
			dbufp = bufinit(tp, ALARM | RESTORE);
			dbufp = sba(dbufp, 0);
			cp = ucmsg;
			while (*cp)
				*dbufp++ = tubotab[*cp++];
			ttp->t_curpos = 80;
			ttp->t_ccws[1].cc_cmd = EWRITE;
			ttp->t_ccws[1].cc_count = dbufp - ttp->t_dbuf;
			if (ttp->t_bits & T_FS) {
				ttp->t_ccws[1].cc_ida = 0;
				ttp->t_ccws[1].cc_addr = (int) ttp->t_dbuf;
			}
			sio(tp->t_addr, ttp->t_ccws + 1, tubrwint, (int) tp);
			ttp->t_bits &= ~T_RIP;
			break;
		default:
			printf("Unrecoverable unit check(s) on tube %x\n", tp->t_addr);
			tp->t_state &= ~(CARR_ON | BUSY);
			ttp->t_resid = -1;
			ttp->t_errct = 0;
			break;
		}
	} else if (csw->cs_de) {
		tp->t_state &= ~BUSY;
		ttp->t_resid = csw->cs_count;
		if ((ttp->t_bits & T_FS) == 0) {   /* normal mode */
			if (ttp->t_bits & T_RIP) {  /* process results of read */
				ttp->t_bits &= ~(T_ATTN | T_RIP);
                                if (ttp->t_bits & T_FSNORM && tp->t_outq.c_cc == 0 && ttp->t_dbuf[0] == CLEAR) {
                                        fsrestor(tp);
                                        wakeup((caddr_t) &ttp->t_bits);
                                } else {
				        tp->t_state |= BUSY;
				        dbufp = tubinput(tp);
				        dbufp = tubochars(tp, dbufp);
				        ttp->t_ccws[1].cc_cmd = WRITE;
				        ttp->t_ccws[1].cc_count = dbufp - ttp->t_dbuf;
				        sio(tp->t_addr, ttp->t_ccws, tubrwint, (int) tp);
				}
			} else if (ttp->t_bits & T_ATTN) {  /* unserviced attention - start a read */
				tp->t_state |= BUSY;
				ttp->t_ccws[1].cc_cmd = READMOD;
				ttp->t_ccws[1].cc_count = BUFLEN;
				sio(tp->t_addr, ttp->t_ccws, tubrwint, (int) tp);
				ttp->t_bits |= T_RIP;
			} else if (tp->t_outq.c_cc)   /* start normal output */
				tubstart(tp);
		} else                             /* full screen mode */
			if (ttp->t_errct > 2) {   /* i/o not done due to uc */
				ttp->t_resid = -1;
				ttp->t_ccws[1].cc_addr = (int) ttp->t_idaws;
				ttp->t_ccws[1].cc_ida = 1;
			}
		ttp->t_errct = 0;
	}
	if ((tp->t_state & BUSY) == 0 && ttp->t_bits & T_ASLEEP)
		wakeup((caddr_t) ttp);
}

/*
 * Called by tubrwint to process input from a tube
 * (i.e., move chars from device buffer to the input queue (through ttyinput)).
 * Also, place response to read in dbuf.
 */
char *
tubinput(tp)
register struct tty *tp;
{
	register struct tube *ttp;
	register char *dbufp, *cp, *redp;
	int i, keynum;

	ttp = tptrs[minor(tp->t_dev)];
	dbufp = NULL;
	keynum = -1;
	switch (ttp->t_dbuf[0]) {     /* switch on AID */

	case CLEAR:
		ttp->t_curpos = 0;
		ttp->t_bits &= ~T_MOREMSG;
		dbufp = bufinit(tp, RESTORE);
		*dbufp++ = IC;
		break;

	case TESTREQ:
		keynum = 0;
		break;

	case PA1:
		ttyinput(CINTR, tp);
		break;

	case PA2:
		ttyinput(CQUIT, tp);
		break;

	case PF1:  case PF2:  case PF3:  case PF4:
	case PF5:  case PF6:  case PF7:  case PF8:
	case PF9:  case PF10: case PF11: case PF12:
		keynum = ttp->t_dbuf[0] & 017;
		break;

	case ENTER:
		keynum = NPFK + 1;
		break;
	}
	if (keynum >= 0) {    /* i.e. enter, pf, or test req key */
		if (keynum == ttp->t_re_back || keynum == ttp->t_re_forw) { /* redisplay key pressed */
			dbufp = bufinit(tp, RESTORE);
			dbufp = ra(dbufp, MSGPOS-1, 0); /* clear input area */
			dbufp = sba(dbufp, MAXPOS+2);
			if (ttp->t_re_off == NRE)
				if (keynum == ttp->t_re_back)
					ttp->t_re_off = 0;
				else
					ttp->t_re_off = 1;
			else
				if (keynum == ttp->t_re_back)
					ttp->t_re_off = (ttp->t_re_off + (NRE - 1)) % NRE;
				else
					ttp->t_re_off = (ttp->t_re_off + (NRE + 1)) % NRE;
			cp = ttp->t_re_buf[(ttp->t_re_idx + ttp->t_re_off) % NRE];
			while (*dbufp++ = *cp++)
				;
			*(dbufp - 1) = IC;
		} else
			ttp->t_re_off = NRE;

		if (keynum == NPFK + 1) {      /* enter key pressed */
			dbufp = &ttp->t_dbuf[6];        /* beginning of data */
			ttp->t_dbuf[BUFLEN - ttp->t_resid] = 0;  /* insure null at end */
			if (ttp->t_dbuf[3] == SBA && *dbufp)
				if (tp->t_flags & ECHO) {
					ttp->t_re_idx = (ttp->t_re_idx + 1) % NRE;
					redp = ttp->t_re_buf[ttp->t_re_idx];
					while (*dbufp)
						ttyinput(tubitab[*redp++ = *dbufp++], tp);
					*redp = '\0';
				} else
					while (*dbufp)
						ttyinput(tubitab[*dbufp++], tp);
			ttyinput(CR, tp);
			dbufp = bufinit(tp, RESTORE);
			*dbufp++ = IC;
			dbufp = ra(dbufp, MSGPOS-1, 0); /* clear out input area */
		}

		/* send pf key data */
		if (keynum != ttp->t_re_back && keynum != ttp->t_re_forw && keynum <= NPFK) {
			cp = ttp->t_pfk[keynum];
			for (i = 0; i < PFLEN && *cp; i++)
				ttyinput(*cp++, tp);
		}
	}
	if (dbufp == NULL)
		dbufp = bufinit(tp, RESTORE);
	return (dbufp);
}

/*
 * Put a set buffer address sequence in the output buffer.
 */
static char *
sba(cp, addr)
register char *cp;
register int addr;
{
	*cp++ = SBA;
	*cp++ = tubecc[addr >> 6];
	*cp++ = tubecc[addr & 077];
	return(cp);
}

/*
 * Put a repeat-to-address sequence in the buffer.
 */
static char *
ra(cp, addr, c)
register char *cp;
register int addr;
char c;
{
	*cp++ = RA;
	*cp++ = tubecc[addr >> 6];
	*cp++ = tubecc[addr & 077];
	*cp++ = c;
	return(cp);
}

/*
 * Put sttribute bytes in buffer
 */
static char *
bufinit(tp, wcc)
register struct tty *tp;
register int wcc;
{
	register char *dbufp;
	register struct tube *ttp;

	ttp = tptrs[minor(tp->t_dev)];
	dbufp = ttp->t_dbuf;
	*dbufp++ = tubecc[wcc];
	/* set up output area */
	dbufp = sba(dbufp, MSGPOS - 1);
	*dbufp++ = SF;
	*dbufp++ = tubecc[PROT];
	/* set up input area */
	dbufp = sba(dbufp, MAXPOS + 1);
	*dbufp++ = SF;
	*dbufp++ = tubecc[MDT + ((tp->t_flags & ECHO) ? 0 : NODISP)];
	/* Leave cursor at input area - caller can IC or whatever. */
	return(dbufp);
}
