#include "../h/param.h"
#include "../h/signal.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/proc.h"
#include "../h/370.h"
#include "../h/reg.h"
#include "../h/stats.h"
#include "../h/bitmap.h"

#define EXTCODE (*(unsigned short *)134)
#define CLKCOMP 0x1004
#define CPUTIMER 0x1005
#define VMCF    0x4001
#define LDSF    0x2401

/*
 * External interrupt handler.
 */
extrn(ar0)
int *ar0;
{
	stats.s_exts++;

	switch(EXTCODE) {

        case VMCF:
                vmcfint();
                break;

        case LDSF:
                ldsfint();
                break;

	case CLKCOMP:
		tick();
		break;

	case CPUTIMER:
                if(ar0[RPS] & UMODE) slice();
		break;
	}
}

/*
 * SCHMAG and SCHUNIT are the magic numbers for feedback scheduling.
 * SCHMAG determines how much recent cpu use determines current priority;
 * SCHUNIT is the unit in which time is measured. (60 for 1/60 sec)
 * If SCHUNIT is too big, p_cpu will grow too rapidly and hit the limit
 * of 255 too quickly. Conversely, if it is too small, there will be too
 * little discrimination based on cpu time.
 */
#define SCHMAG  8/10
#define SCHUNIT 60

/*
 * Once per second clock tick.
 * Set up next interrupt, update system time,
 * and update process priorities based on ratio of process
 * time over UNIX virtual time.
 */
tick()
{
	tod_t tod, a_stck();
	static cpu_t lasttick;
	register struct proc *p;
	register int a, deltat;
	int t, itime;

	tod = a_stck();                 /* get TOD */
	a_sckc(tod + (1000000L<<12));   /* set comparator to 1 second later */
	itime = ctime(tod);
	/*
	 * Update global system time and find how long since we were last here
	 */
	t = itime - time;
	time = itime;

	/*
	 * Get elapsed virtual real time from stats.s_ttime.
	 * This is real time minus involuntary wait time.
	 */
	deltat = (stats.s_ttime - lasttick) >> 12;
	lasttick = stats.s_ttime;

	/*
	 * Scan processes, updating priorities and sending alarm signals.
	 */
	for(p = &proc[0]; p < &proc[NPROC]; p++)
	if(p->p_stat && p->p_stat < SZOMB) {
		a = (int)(p->p_time >> 12);
		p->p_time = 0L;
		a = p->p_cpu + SCHUNIT * a / deltat;
		a = a * SCHMAG + p->p_nice - NZERO;
		if(a < 0)
			a = 0;
		else if(a > 255)
			a = 255;
		p->p_cpu = a;
		if(p->p_pri >= PUSER)
			setpri(p);
		if(p->p_clktim) {
			if((p->p_clktim -= t) <= 0) {
				p->p_clktim = 0;
				psignal(p, SIGALRM);
			}
		}
	}
	iotic();
	runrun++;
}
 
/*
 * End of user time slice.
 * Bump user priority and poke scheduler.
 * May generate a signal.
 */
slice()
{
	struct proc *p;

	p = u.u_procp;
	if(p->p_pri < 127)
		p->p_pri++;
	runrun++;
        if ((u.u_procp->p_flag & STRC) && btst(u.u_sigtr, SIGSLICE)) {
		psignal(u.u_procp, SIGSLICE);
		issig();
		bclr(u.u_sig, SIGSLICE);
	}
}

/*
 * Convert 370 TOD format to UNIX time
 */
ctime(tod)
tod_t tod;
{
	register int t;
	static int dstflag = -1;

	/*
	 * Convert to seconds since May 11, 1971 at 11:56:54
	 * See 370 POOP
	 */
	t = ((tod & 0x7FFFFFFFFFFFFFFFL) >> 12) / 1000000;
	/*
	 * Add in seconds between Jan 1, 1970 and May 11, 1971 at 11:56:54
	 */
	t += 42811014;
	/*
	 * If TOD clock runs in local time, add in time zone to make it GMT
	 */
	if(LOCALTOD) {
		t += TIMEZONE*60;
		if(DSTFLAG) {
			if((t%3600) == 0 || dstflag == -1)
				dstflag = isdst(t);
			if(dstflag) t -= 3600;
		}
	}
	return(t);
}

#define dysize(y) ((y%4) == 0 ? 366 : 365)

/*
 * Determine if tim is in daylight savings or not
 * so we can un-adjust clock if in local time
 * This code is stolen and compressed from ctime(2)
 * Called once an hour from ctime above
 */
isdst(tim)
{
	register int day, y, d;
	int hour, wday, daylb, dayle;

	hour = tim / 3600;
	day = hour / 24;
	hour = hour % 24;
	wday = (day+7340036) % 7;
	for(y=70; day >= dysize(y); y++)
		day -= dysize(y);
	d = 119 + dysize(y) - 365;
	daylb = d - (d - day + wday + 700) % 7; /* last sun in apr */
	d = 303 + dysize(y) - 365;
	dayle = d - (d - day + wday + 700) % 7; /* last sun in oct */
	if((day > daylb || (day == daylb && hour >= 2)) &&
	   (day < dayle || (day == dayle && hour < 1)))
		return(1);
	return(0);
}
