/*
 * ss preprocessor
 */

/*
 * This preprocessor takes some of the ss and mm commands (listed later)
 * and replaces them with a few nroff commands.  Below are descriptions
 * of the number and string registers and the diversions output by this
 * program.
 *
 * number registers
 * 1a-z  1a contains the number of type 0a figures ...
 * i1-9  i1 contains the distance between the left margin and the text for the level 1 (most major) list ...
 * j1-9  j1 contains the distance between the left margin and the label for the level 1 list when l_lps is 0 ...
 * l1-9  l1 contains the item number (letter) of the current item in the level 1 list ...
 * ue    saves current adjustment mode when processing figures
 * uf    saves current filling mode when processing figures
 * uh    used to create the current label by .sl
 * wd    contains the number of characters in the current label, then the number of spaces between the label and the text
 * we    contains the distance between the left margin and the current label when l_lps is> 0
 * wk    contains the current appendix letter
 * wu    contains the number of unique figure types
 * ww    a letter used for figure diversions, figure names and figure counts
 *
 * string registers
 * 0a-z  a string for each type of figure, for ex. 0a might be "Figure", 0b "Example" ...
 * js    contains the label for the current list entry
 *
 * diversions
 * 2a-z  2a contains all the table of "figures" entries for type 0a figures ...
 */

#include <stdio.h>
#include <stdioerr.h>
#include <ctype.h>
#include <sys/types.h>
#include <time.h>

/* these are the commands that are preprocessed */
#define AL 1
#define AP 2
#define BL 3
#define CM 4
#define DE 5
#define DL 6
#define DS 7
#define FG 8
#define FO 9
#define GF 10
#define HD 11
#define IL 12
#define LE 13
#define LX 14
#define NR 15
#define OL 16
#define RL 17
#define SF 18
#define SH 19
#define SL 20
#define SO 21
#define VL 22

/* these characters become part of name in the TBL structure to identity which kind of name it is */
#define NRCHAR 1   /* number register */
#define DSCHAR 2   /* string register */
#define DECHAR 3   /* user defined macro */
#define CMDCHAR 4  /* preprocessed command */

#define MAXARGS 100     /* maximum number of arguments */
#define ARGBUFLEN 1000  /* maximum number of characters in arguments */
#define MACBUFLEN 1000  /* maximum number of characters in macro */
#define MAXFIGS 10      /* maximum number of figure types */
#define MXLSTNST 9      /* maximum depth of nested lists (9 is absolute maximum because of i1-9, j1-9 and l1-9 number registers) */
#define LIND 5          /* standard list indent */
#define RLIND 7         /* roman list indent */

#define strdup(s)       (strcpy(malloc(strlen(s)+1), s))
char *strcpy();
char *malloc();

FILE *infil;  /* the current input file */
FILE *tmp;    /* a memory file where most of output of press is stored temporarily */

FILE *fil_stk[20];         /* a file stack for .so's */
FILE **fsp = &fil_stk[0];  /* a pointer to the current file on the stack */

char *fil_nam[20];         /* a stack of input file names (for error messages only) */
char **fnp = &fil_nam[0];  /* a pointer to the current file name */

#define HSHSIZ 401  /* the size of the hash table (TBL) */
#define hash(typ, str)  ((typ << 3) ^ str[0] ^ str[1]) % HSHSIZ  /* the hashing algorithm */

typedef union val {  /* contains the value of an entry in the hash table */
        int cmdnum;  /* a command number for preprocessed commands or */
        char *regvalue;  /* the value of the number or string register or macro */
} VAL;

typedef struct tbl {   /* the hash table */
	char name[4];  /* the name to be hashed with the proper character (NRCHAR ...) prepended */
	VAL value;     /* the value (see VAL) */
	struct tbl *chain;  /* a pointer to the next entry if there is a hashing conflict */
} TBL;

TBL *hashp[HSHSIZ];  /* hash table pointers */

char argbuf[ARGBUFLEN];  /* contain the arguments to the current command */
char *arg[MAXARGS];      /* pointers to the arguments, ignore arg[0] */

int ln_cnt;             /* the number of the current input line of the current file */
int ln_cnt_stk[20];     /* the current ln_cnt when a file is pushed on the stack because of a .so (for error messages only) */
int *lsp = ln_cnt_stk;  /* a pointer to the ln_cnt stack */

int lin_len = 61;  /* the line length (61 is the default) */

int lst_lev = 0;  /* current list level */
int ol_lev  = 0;  /* current outline list level */

struct lst_stk {      /* list characteristics */
	int t_ind;    /* text indent (from the current left margin) */
	int l_ind;    /* label indent (the number of spaces from the current left margin that the label will start in most cases) */
	int l_lps;    /* label left pad size (if this is > 0 this many blanks will separate the label and the text) */
	int l_typ;    /* label type: 0 for a string which is in label, 2 for [], 3 for ), 4 for variable labels */
	char *label;  /* label (null, a string, 1, A, a, I or i) */
	int le_sp;    /* list entry space: 1 if there is a blank line before entries, 0 if not */
	int itm_num;  /* number of items currently in this list */
} lst_stk[MXLSTNST+1];   /* lst_stk[0] is not used */

struct lst_stk *sp;  /* a pointer to the lst_stk currently of interest */

struct figs {        /* stores figure types and numbers */
	char *type;
	int count;
} figs[MAXFIGS];

int pos;           /* position in figs of current type of figure */
int emptyfig = 0;  /* first empty entry in figs */

int chd_lev;     /* the current heading level */
int sec[6];      /* one for each heading level, contains the current number each level is at, sec[0] not used */
char hdnum[50];  /* a string containing the current section heading number */
int app_num;     /* the number of the current appendix */

int inmac = 0;  /* 1 when processing a user defined macro call, 0 otherwise */

char boldc = '\0';  /* bold character from .fo */
char italc = '\0';  /* italic character from .fo */
int bfont = -1;     /* 1 when in bold, -1 when not */
int ifont = -1;     /* 1 when in italic, -1 when not */

char buff[BUFSIZ];  /* buffer for dumping stuff in memory file */

int past_eof();
char *pd_sec_hd();
char *exp_reg();
TBL *tblinsrt();
TBL *tbllkup();

main(argc, argv)
int argc;
char **argv;
{
	char cmd[3];
	register int c;
	register int i;

	for (i=1; argv[i][0] == '-'; i++, argc--, argv++)  /* interpret '-' arguments to command */
		switch (argv[i][1]) {
		case 'p' :  if (argv[i][2] != '\0')
		                    lin_len = 6.1 * atoi(&argv[i][2]) + .5;
			    break;
		default  :  infil = stdin;  /* kludge for error func. */
		            error("%s is an invalid option", *argv[i]);
		}
	for (i=0; i<MAXFIGS; i++) {  /* initialize figs structure */
		figs[i].type = "";
		figs[i].count = 0;
	}
	tmp = memopen("press temp file", "w"); /* for most of output */
	entr_cmd();  /* put commands to be preprocessed in hash table */
	entr_str();  /* put pre-defined strings in hash table */
	printf(".af wk A\n");
	fprintf(tmp, ".af ww a\n");
	if (argc > 1)  /* open first file */
		for (argc--; argc>0; argc--)
			if ((infil=fopen(*++argv, "r")) != NULL) {
				*fnp= *argv;
				break;
			} else {
				infil = stdin;
				error("cannot open file %s", *argv);
			}
	else
	        infil = stdin;
	onerror (EPASTEOF, past_eof);
	while (argc > 0 ) {  /* read and process all the input */
                while ((c=getc(infil)) != EOF) { /*assumes getting char in col 1 */
			if (bfont > 0 || ifont > 0) {
				error("line %d ended in bold or italics", ln_cnt);
				bfont = -1;
				ifont = -1;
				fprintf(tmp, ".ft R\n");
			}
			if (!inmac)
			        ln_cnt++;
                        if (c != '.')
                                skipln(c);
                        else {
                                cmd[0] = getc(infil);
                                cmd[1] = getc(infil);
				if (cmd[1] == '\n') {
					ungetc('\n', infil);
					cmd[1] = '\0';
				}
                                cmd[2] = '\0';
                                process(cmd);
                        }
                }
		if (!inmac)
	                fclose(infil);
		else
			memclose(infil);
		if (fsp > &fil_stk[0]) {  /* switch from .so file back */
			infil = *--fsp;
			if (!inmac) {
			        ln_cnt = *--lsp;
			        fnp--;
			} else
				inmac = 0;
	        } else  /* get the next file */
                        for (argc--; argc>0; argc--)
                                if ((infil=fopen(*++argv, "r")) != NULL) {
                                        *fnp= *argv;
                                        ln_cnt = 0;
                                        break;
                                } else {
				        infil = stdin;
                                        error("cannot open file %s", *argv);
				}
	}
	while (lst_lev-- > 0)  /* there were missing .lx's at the end */
		error("a missing .lx was detected at line %d", ln_cnt);
        memreopen(tmp, "r");  /* dump out the saved stuff */
	onerror (EPASTEOF, EIGNORE);
        while ((c = fread(buff, 1, BUFSIZ, tmp)) > 0)
		fwrite(buff, 1, c, stdout);
        printf(".nr wu %d\n.em\n", emptyfig);
}

/*
 * Output to tmp c and the rest of the line (do the next line too
 * if the newline is escaped).  Check each character to see if it
 * is a font change character and if so, insert the proper \f
 * sequence.
 */
skipln(c)
register int c;
{
	for (; c != '\n' ; c=getc(infil))
		if (c == '\\')
			if ((c=getc(infil)) == boldc || c == italc)  /* output the character without the backslash */
				putc(c, tmp);
			else {  /* output a backslash and the next character (even if a newline) and continue getc's */
	                        putc('\\', tmp);
	                        putc(c, tmp);
			        if (c == '\n' && !inmac)
				        ln_cnt++;
			}
		else if (c == boldc) {
                        putc('\\', tmp);
                        putc('f', tmp);
			if (bfont > 0)
				putc('P', tmp);
			else
				putc('B', tmp);
			bfont = -bfont;
		} else if (c == italc) {
                        putc('\\', tmp);
                        putc('f', tmp);
			if (ifont > 0)
				putc('P', tmp);
			else
				putc('I', tmp);
			ifont = -ifont;
		} else
	                putc(c, tmp);
	putc(c, tmp);
}

/*
 * A command has been recognized.  Check to see if it is a command
 * which must be pre-processed, a user defined macro or a command
 * which is passed on to ss.
 */
process(cmd)
char *cmd;
{
	TBL *t;

        if ((t=tbllkup(CMDCHAR, cmd)) == NULL)  /* it's not a preprocessed command */
		if ((t=tbllkup(DECHAR, cmd)) == NULL) {  /* it's not a user defined macro */
                        fprintf(tmp, ".%s", cmd);  /* pass the command through with noprocessing */
                        skipln(getc(infil));
                } else   /* it is a user defined macro */
			get_mac(t);
	else  /* it is a preprocessed ss command */
                switch (t->value.cmdnum) {
                case AL:  do_al();     break;
                case BL:  do_bl();     break;
                case CM:  do_cm();     break;
                case DL:  do_dl();     break;
                case IL:  do_il();     break;
                case OL:  do_ol();     break;
                case RL:  do_rl();     break;
                case VL:  do_vl();     break;
                case LE:  do_le();     break;
                case LX:  do_lx();     break;
                case HD:  do_hd();     break;
                case AP:  do_ap();     break;
                case GF:  do_fig(GF);  break;
                case FG:  do_fig(FG);  break;
                case DS:  do_reg(DS);  break;
                case NR:  do_reg(NR);  break;
                case SH:  do_sh();     break;
                case SF:  do_sf();     break;
                case SL:  do_sl();     break;
                case SO:  do_so();     break;
                case DE:  do_de();     break;
		case FO:  do_fo();     break;
                }
}

/*
 * Get the arguments to a command.  Certain character sequences
 * must be recognized and processed specially as noted below.  The
 * number of arguments found is returned.
 */
get_args()
{
	register int c = '\0';
	int lastc;
        int i          = 1;
	int newline    = 0;   /* 1 if an unescaped \n was found */
	int newarg     = 1;   /* is 1 from when the end of an argument was recognized until the first character of the next is found */
	int instring   = -1;  /* 1 when in a double quoted string, -1 when not */
	int arg_cnt;
	int type;  /* DSCHAR or NRCHAR */
	register char *p = argbuf;
	char buf[3];
	char *b = buf;
	TBL *tp;

        while (!newline) {  /* process until an unescaped newline is found */
                lastc = c;
                c = getc(infil);
                if (newarg == 1 && c != ' ') {  /* the first character of a new argument has been found */
			if (i > MAXARGS) {  /* ignore rest of line after MAXARGS arguments have been found*/
				error("more than %d arguments were given to the command on line %d", MAXARGS, ln_cnt);
				while (c != '\n' || c == '\n' && lastc == '\\') {
					lastc = c;
					c = getc(infil);
				}
				newline = 1;
			} else {  /* start a new arg */
                                arg[i++] = p;
                                newarg = 0;
			}
                }
		if (c == '"' && lastc == '\\') {  /* a \" has been found, ignore the rest of the line */
			p--;
			while ((c=getc(infil)) != '\n')
				;
			newline = 1;
		} else if ((c == '*' || c == 'n') && lastc == '\\') {  /* a \* or \n has been found, substitute the string or number */
			if (c == '*')
				type = DSCHAR;
			else
				type = NRCHAR;
	                if ((c=getc(infil)) == '(') {
	                        *b++ = getc(infil);
	                        *b++ = getc(infil);
	                        *b   = '\0';
	                } else {
	                        *b++ = c;
	                        *b   = '\0';
	                }
		        if ((tp=tbllkup(type, buf)) == NULL) {
				if (type == DSCHAR)
					*p++ = '*';
				else
					*p++ = 'n';
				if (strlen(buf) == 2)
					*p++ = '(';
				strcpy(p, buf);
				p += strlen(buf);
			} else {
			        p--;
			        strcpy(p, tp->value.regvalue);
			        p += strlen(tp->value.regvalue);
			}
                } else if (c == '\n')  /* a newline was found, check to see if it wasescaped */
                        if (lastc == '\\') {
				p--;
				if (p == arg[i-1]) {
					newarg = 1;
					i--;
				}
				if (!inmac)
				        ln_cnt++;
			} else
                                newline = 1;
                else if (c == ' ' && newarg == 0)  /* a blank was found at the end of or in an argument */
                        if (lastc == '\\' || instring == 1) /* if it was escaped or it is part of a double quoted argument then it is part of the argument */
                                *p++ = c;
                        else {  /* it marks the end of an argument */
                                newarg = 1;
                                *p++ = '\0';
                        }
                else if (c == '"') {  /* toggle instring (instring is 1 if in a double quoted string, -1 if not) */
                        instring = -instring;
			if (lastc == '"') {  /* a null string, make it a \& */
				*p++ = '\\';
				*p++ = '&';
			}
		} else if (c == boldc && lastc != '\\') {  /* if the bold font change character was found insert the proper \f sequence */
                        *p++ = '\\';
                        *p++ = 'f';
			if (bfont > 0)
				*p++ = 'P';
			else
				*p++ = 'B';
			bfont = -bfont;
		} else if (c == italc && lastc != '\\') {  /* if the italic font change character was found insert the proper \f sequence */
                        *p++ = '\\';
                        *p++ = 'f';
			if (ifont > 0)
				*p++ = 'P';
			else
				*p++ = 'I';
			ifont = -ifont;
                } else if (c != ' ')  /* add the character to the current argument if it isn't a blank (if it is a blank it is the 2nd or following blank between arguments) */
                        *p++ = c;
                if (p >= &argbuf[ARGBUFLEN-4]) {  /* it's possible that 4 characters could be added before this ischecked again */
                        error("too many characters in arguments to command on line %d", ln_cnt);
			while (c != '\n' || c == '\n' && lastc == '\\') {  /* ignore the rest of the line */
				lastc = c;
				c = getc(infil);
				}
			newline = 1;
                        }
        }
	if (arg[i-1] == p)  /* the last "argument" started with \" */
		arg_cnt = i - 2;
	else
	        arg_cnt = i - 1;
	*p++ = '\0';  /* put a '\0' at the end of the last argument */
        while (i < MAXARGS && p < &argbuf[ARGBUFLEN-1]) {  /* make the rest of the arguments '\0' */
		arg[i++] = p;
	        *p++ = '\0';
	}
	return(arg_cnt);
}

/*
 * Process a .al command.
 */
do_al()
{
	char a1;  /* shorthand for *arg[1] */

        if (get_args() > 2)
		error("the .al on line %d has an incorrect number of arguments", ln_cnt);
        if (lst_lev == MXLSTNST) {
                error("lists may nest only %d deep, the .al on line %d was ignored", MXLSTNST, ln_cnt);
		return;
	}
	sp = &lst_stk[++lst_lev];
	a1 = *arg[1];
        if (a1 != '\0' && a1 != '1' && a1 != 'A' &&
            a1 != 'a' && a1 != 'I' && a1 != 'i') {
                error("the .al on line %d contains an invalid argument", ln_cnt);
		arg[1] = "1";
	}
        if (a1 == 'I' || a1 == 'i')
                sp->t_ind = RLIND;
        else
                sp->t_ind = LIND;
        sp->l_ind = 0;
        sp->l_lps = 2;
        sp->l_typ = 1;
        if (a1 == '\0')
                sp->label = "1";
        else
                sp->label = strdup(arg[1]);
        if (*arg[2] == '\0')
                sp->le_sp = 1;
        else
                sp->le_sp = 0;
	init_lst();
}

/*
 * Process a .ap command.
 */
do_ap()
{
	register int i;
	register char *p;
	int arg_cnt;

	while (lst_lev > 0) {
		error("a missing .lx was detected at line %d", ln_cnt);
		ungetc('\n', infil);  /* this will be the "argument" to the do_lx */
		do_lx();
	}
	arg_cnt = get_args();
	fprintf(tmp, ".sk\n.fi\n.na\n.up 999\n.in 15\n.ti 0\n.if \\n7 \\(bs\\c\n");
	fprintf(tmp, ".nr wk %d\n\\fIAppendix\\ \\n(wk.\\ \\ \\ \\ \\c\n\\%%", ++app_num);
	for (i=1, p=argbuf; i<=arg_cnt; p++)  /* output each argument with a \% at the beginning */
                if (*p == '\0') {
			if (i++ < arg_cnt) {
	                        putc(' ', tmp);
	                        putc('\\', tmp);
	                        putc('%', tmp);
	                        }
                } else
                        putc(*p, tmp);
	fprintf(tmp, "\\fP\n.br\n.up 0\n.xf 1\n.dm ze\n.sp\n.ti 0\n");
        fprintf(tmp, "Appendix\\ \\n(wk.\\ \\ \\ \\ \\c\n");
        tc_entry(15, 1, arg_cnt, AP); /* indent 15, arg[1] first */
	fprintf(tmp, ".ad\n.in 0\n");
	chd_lev = 0;  /* reset regular heading levels */
	for (i=1; i<=5; i++)
		sec[i] = 0;
}

/*
 * Process a .bl command.
 */
do_bl()
{
        if (get_args() > 1)
		error("the .bl on line %d has an incorrect number of arguments", ln_cnt);
        if (lst_lev == MXLSTNST) {
                error("lists may nest only %d deep, the .bl on line %d was ignored", MXLSTNST, ln_cnt);
		return;
	}
	sp = &lst_stk[++lst_lev];
        sp->t_ind = LIND;
        sp->l_ind = 2;
        sp->l_lps = 0;
        sp->l_typ = 0;
        sp->label = "\\(bu";
        if (*arg[1] == '\0')
                sp->le_sp = 1;
        else
                sp->le_sp = 0;
	init_lst();
}

/*
 * Process (throw away) a .cm command.
 */
do_cm()
{
	while (getc(infil) != '\n')
		;
}

/*
 * Process a .de command.  This does not exactly reproduce the nroff
 * .de command, but instead does the minimum amount of processing
 * necessary to process macros defined by users.  This is defined to
 * be replacing each pair of backslashes with a single backslash.
 * The macro definition is then stored away in the hash table.
 * It is also output (exactly as input except for comments) for use
 * by nroff if necessary.
 */
do_de()
{
	register int c     = '\0';
	register int lastc = '\0';
	int de_ln_cnt;
	int prevc;
	char buf[MACBUFLEN];  /* temporary storage area for the macro definition */
	register char *p = buf;
	TBL *tp;

	if (get_args() > 1)
		error("the .de on line %d has an incorrect number of arguments", ln_cnt);
	fprintf(tmp, ".de %s\n", arg[1]);
	tp = tblinsrt(DECHAR, arg[1]);
	de_ln_cnt = ln_cnt;
	for (;;) {
		if (p > buf + MACBUFLEN-2)
			error("incorrect .de at line %d, missing .., or definition too large", de_ln_cnt);
		prevc = lastc;
		lastc = c;
		putc((c=getc(infil)), tmp);
		if (c == '\n' && !inmac)
			ln_cnt++;
		if (c == '.' && lastc == '.' && prevc == '\n') {  /* the end of the definition was found */
			*--p = '\0';
			while ((c=getc(infil)) != '\n')  /* ignore the rest of the line */
				;
			putc('\n', tmp);
			if (!inmac)
			        ln_cnt++;
                        break;
		} else if (c == '\\' && lastc == '\\')  /* a pair of backslashes were found */
			c = '\0';  /* ignore the second \ and make the first \ invisible */
		else if (c == '"' && lastc == '\\' && prevc != '\\') {  /* an unescaped comment was found */
			if (prevc == '.') {    /* a . preceded the \". */
				*(p-1) = ' ';  /* just in case the . was in column 1, put two blanks after it so a command named .newline isn't called */
				*p++ = ' ';
				*p++ = '\n';
			} else
			        *(p-1) = '\n';
			while ((c=getc(infil)) != '\n')  /* ignore the rest of the line */
				;
			putc('\n', tmp);
			if (!inmac)
			        ln_cnt++;
		} else
			*p++ = c;
	}
	tp->value.regvalue = strdup(buf);
}

/*
 * Process a .dl command.
 */
do_dl()
{
        if (get_args() > 1)
		error("the .dl on line %d has an incorrect number of arguments", ln_cnt);
        if (lst_lev == MXLSTNST) {
                error("lists may nest only %d deep, the .dl on line %d was ignored", MXLSTNST, ln_cnt);
		return;
	}
	sp = &lst_stk[++lst_lev];
        sp->t_ind = LIND;
        sp->l_ind = 2;
        sp->l_lps = 0;
        sp->l_typ = 0;
        sp->label = "-";
        if (*arg[1] == '\0')
                sp->le_sp = 1;
        else
                sp->le_sp = 0;
	init_lst();
}

/*
 * Process .fg and .gf commands.
 */
do_fig(cmd)
{
	register int i;
	register char *p;
	int arg_cnt;
	int foundflg = 0;  /* 1 when a figure type has been found in figs */
	int tit_len  = 0;  /* total length of figure title */
	int blanks;        /* the number of blanks between the . after the number and the first word of the title */

        arg_cnt = get_args();
	if (cmd == FG) {
                for (i= ++arg_cnt; i>1; i--)
                        arg[i] = arg[i-1];
                arg[1] = "Figure";
	}
	fprintf(tmp, ".nr ue \\n(.j\n.nr uf \\n(.u\n");
	for (i=0; i<emptyfig && !foundflg; i++)  /* try to find the figure type in figs */
		if (strcmp(figs[i].type, arg[1]) == 0) {
			foundflg = 1;
			pos = i;
		}
	if (!foundflg) {  /* it wasn't found in figs so enter it */
		figs[emptyfig].type = strdup(arg[1]);
		figs[emptyfig].count = 1;
		pos = emptyfig++;
	} else
		figs[pos].count++;
	fprintf(tmp, ".nr ww %d\n.nr 1\\n(ww +1\n", pos+1); /* figs starts at 0, but ww must start at 1 so add 1 to pos */
	if (!foundflg)
		fprintf(tmp, ".ds 0\\n(ww %s\n", arg[1]);
	if (figs[pos].count < 10)
		blanks = 4;
	else
		blanks = 3;
	fprintf(tmp, ".xf 1\n.na\n.fi\n.in %d\n.ti 0\n", strlen(figs[pos].type) + 7);
	for (i=2; i<=arg_cnt; i++)
		tit_len += strlen(arg[i]) + 1;  /* argument plus a blank */
	if (tit_len < lin_len)
		fprintf(tmp, ".ce\n");
        fprintf(tmp, "\\fI%s\\ %d.\\fR\\ \\ \\ \\c\n",
		figs[pos].type, figs[pos].count);
	if (blanks == 4) {  /* add an extra blank before the title */
		putc('\\', tmp);
		putc(' ', tmp);
	}
	putc('\\', tmp);
	putc('%', tmp);
	if (cmd == FG)
		p = argbuf;
	else /* cmd == GF */
		p = &argbuf[strlen(arg[1])+1];
	for (i=1; i<arg_cnt; p++) /* put out the arguments character by character, separated by ' \%' */
                if (*p == '\0') {
			if (i < arg_cnt - 1) {
	                        putc(' ', tmp);
	                        putc('\\', tmp);
	                        putc('%', tmp);
			}
                        i++;
                } else
			putc(*p, tmp);
	fprintf(tmp, "\n.xf 1\n.dm 2\\n(ww\n.sp\n.ti 0\n");
        if (blanks == 3)
                fprintf(tmp, "\\fB%s\\ %d.\\ \\ \\ \\c\n", figs[pos].type,
                        figs[pos].count);
        else  /* blanks == 4 */
                fprintf(tmp, "\\fB%s\\ %d.\\ \\ \\ \\ \\c\n", figs[pos].type,
                        figs[pos].count);
        tc_entry(strlen(figs[pos].type) + 7, 2, arg_cnt, cmd); /* start at arg[2] */
	fprintf(tmp, ".in\n.ad \\n(ue\n.if !\\n(uf .nf\n");
}

/*
 * Process a .fo command.
 */
do_fo()
{
	register int c;

	boldc = '\0';
	italc = '\0';
	while ((c=getc(infil)) == ' ')
		;
	if (c != '\n') {
	        boldc = c;
	        while ((c=getc(infil)) == ' ')
		        ;
	        if (c != '\n') {
	                italc = c;
			while ((c=getc(infil)) != '\n')
				;
		}
        }
}

/*
 * Process a .hd command.
 */
do_hd()
{
	register int i;
	register char *p;
	int arg_cnt;
	int phd_lev;  /* previous heading level */
	int hdlen;    /* indent for section heading text */
	char *hdstr;  /* the section heading number plus blanks */

	while (lst_lev > 0) {
		error("a missing .lx was detected at line %d", ln_cnt);
		ungetc('\n', infil);
		do_lx();
	}
	fprintf(tmp, ".if \\n(vj .fe\n.if \\n(vy .ed\n");
	arg_cnt = get_args();
        if (*arg[1] == '\0' || strcmp(arg[1], "\\&") == 0) {
		error("the .hd on line %d is missing the heading level", ln_cnt);
		arg[1] = "5";
	}
	phd_lev = chd_lev;
	if (strlen(arg[1]) > 2 ||
            strlen(arg[1]) == 1 && isdigit(arg[1][0]) == 0 ||
            strlen(arg[1]) == 2 && isdigit(arg[1][1]) == 0) {
		error("the .hd on line %d contains an invalid heading level", ln_cnt);
		chd_lev = 5;
	}
	if (arg[1][0] == '+')
		chd_lev = chd_lev + atoi(&arg[1][1]);
	else if (arg[1][0] == '-')
		chd_lev = chd_lev - atoi(&arg[1][1]);
	else if (strlen(arg[1]) == 1)
		chd_lev = atoi(arg[1]);
	else {
		error("the .hd on line %d contains an invalid heading level", ln_cnt);
		chd_lev = 5;
	}
	if (chd_lev >= 6) {
		error("the heading level on line %d is too large", ln_cnt);
		chd_lev = 5;
	}
	switch (chd_lev) {
	case 0: sec[1] = 0;
	case 1: sec[2] = 0;
	case 2: sec[3] = 0;
	case 3: sec[4] = 0;
	case 4: sec[5] = 0;
	}
	sec[chd_lev]++;
	if (chd_lev == 0) {
		hdlen = 0;
		*hdnum = '\0';
	} else if (chd_lev == 1) {
		hdlen = 6;
		sprintf(hdnum, "%d.", sec[1]);
	} else if (chd_lev == 2) {
		hdlen = 9;
		sprintf(hdnum, "%d.%d", sec[1], sec[2]);
	} else if (chd_lev == 3) {
		hdlen = 12;
		sprintf(hdnum, "%d.%d.%d", sec[1], sec[2], sec[3]);
	} else if (chd_lev == 4) {
		hdlen = 15;
		sprintf(hdnum, "%d.%d.%d.%d", sec[1], sec[2], sec[3],
			sec[4]);
	} else if (chd_lev == 5) {
		hdlen = 18;
		sprintf(hdnum, "%d.%d.%d.%d.%d", sec[1], sec[2],
			sec[3], sec[4], sec[5]);
	}
	if (chd_lev <= 1)
		fprintf(tmp, ".xf 6\n");
	else
		fprintf(tmp, ".xf 2\n");
	fprintf(tmp, ".ne 4v\n.fi\n.na\n.in %d\n.ti 0\n.if \\n7 \\(bs\\c\n", hdlen);
	hdstr = pd_sec_hd(hdnum, hdlen);
	fprintf(tmp, ".up 999\n%s\\fI\\c\n\\%%", hdstr);
	for (i=2, p = &argbuf[strlen(arg[1])+1]; i<=arg_cnt; p++)
                if (*p == '\0') {
			if (i++ < arg_cnt) {
	                        putc(' ', tmp);
	                        putc('\\', tmp);
	                        putc('%', tmp);
			}
                } else
                        putc(*p, tmp);
        fprintf(tmp, "\\fP\n.br\n.up 0\n.xf 1\n.dm ze\n");
        if (chd_lev <= 1)
                fprintf(tmp, ".sp\n");
        if (chd_lev > 1 && chd_lev != phd_lev)
                fprintf(tmp, ".sp\n");
        fprintf(tmp, ".ti -%d\n", hdlen);
        if (chd_lev <= 1)
                fprintf(tmp, "\\fB\\c\n");
        fprintf(tmp, "%s\\c\n", hdstr);
        tc_entry(hdlen, 2, arg_cnt, HD);  /* start with arg[2] */
	fprintf(tmp, ".ad\n.in 0\n");
	if (chd_lev == 1) /* remove . at end of hdnum (for .sh) */
		hdnum[strlen(hdnum)-1] = '\0';
}

/*
 * Process a .il command.
 */
do_il()
{
        if (get_args() > 2)
		error("the .il on line %d has an incorrect number of arguments", ln_cnt);
        if (lst_lev == MXLSTNST) {
                error("lists may nest only %d deep, the .il on line %d was ignored", MXLSTNST, ln_cnt);
		return;
	}
	sp = &lst_stk[++lst_lev];
        sp->t_ind = strlen(arg[1]) + 4;
        sp->l_ind = 2;
        sp->l_lps = 0;
        sp->l_typ = 0;
	if (*arg[1] == '\0')
		sp->label = "\\&";
	else
	        sp->label = strdup(arg[1]);
        if (*arg[2] == '\0')
                sp->le_sp = 1;
        else
                sp->le_sp = 0;
	init_lst();
}

/*
 * Process a .le command.
 */
do_le()
{
	if (get_args() > 1)
		error("the .le on line %d has an incorrect number of arguments", ln_cnt);
	if (lst_lev == 0) {
		error("no list has been initialized for the .le on line %d", ln_cnt);
		return;
	}
	sp = &lst_stk[lst_lev];
	fprintf(tmp, ".fi\n");
	if (sp->le_sp == 1)
		fprintf(tmp, ".xf 1\n");
	fprintf(tmp, ".in \\n(i%d\n.nr l%d +1\n", lst_lev, lst_lev);
	sp->itm_num++;
        if (*arg[1] != '\0' && sp->l_typ != 4)
                fprintf(tmp, ".ds js %s\n", arg[1]);
	else if (sp->l_typ == 1)
		fprintf(tmp, ".ds js \\n(l%d.\n", lst_lev);
        else if (sp->l_typ == 2)
		fprintf(tmp, ".ds js [\\n(l%d]\n", lst_lev);
	else if (sp->l_typ == 3)
		fprintf(tmp, ".ds js (\\n(l%d)\n", lst_lev);
	else if (sp->l_typ == 4)
		if (*arg[1] != '\0') {
	                fprintf(tmp, ".ds js %s\n", arg[1]);
                        sp->label = strdup(arg[1]);  /* do this for .sl of variable label lists */
		} else {  /* no label was given to variable label list so use a blank */
                        fprintf(tmp, ".ds js \" \n");
                        sp->label = " ";
		}
	else
		fprintf(tmp, ".ds js %s\n", sp->label);
	fprintf(tmp, ".nr wd \\w\\|\\*(js\\|/\\w\\| \\|\n");
	if (sp->l_lps > 0) {
		fprintf(tmp, ".nr we \\n(i%d-%d-\\n(wd\n.if \\n(we<0 .in +(0-\\n(we)\n.ie \\n(we<0 .ti 0\n", lst_lev, sp->l_lps);
	        fprintf(tmp, ".el .ti \\n(we\n.ie \\n(we<0 .nr wd %d\n.el .nr wd \\n(i%d-\\n(we-\\n(wd\n", sp->l_lps, lst_lev);
	} else
	        fprintf(tmp, ".ti \\n(j%d\n.nr wd \\n(i%d-\\n(j%d-\\n(wd\n", lst_lev, lst_lev, lst_lev);
	fprintf(tmp, ".if \\n(wd<=0 .nr wd \\w\\|   \\|/\\w\\| \\|\n");
        fprintf(tmp, ".if !\\|\\*(js\\|\\| \\&\\*(js\\h\\|\\n(wd\\|\\&\\c\n");
}

/*
 * Process a .lx command.
 */
do_lx()
{
	if (get_args() > 1)
		error("the .lx on line %d has an incorrect number of arguments", ln_cnt);
	if (lst_lev == 0) {
		error("an extra .lx was found on line %d", ln_cnt);
		return;
	}
	sp = &lst_stk[lst_lev];
	fprintf(tmp, ".in \\n(i%d-%d\n", lst_lev, sp->t_ind);
	lst_lev--;
	if (*arg[1] == '\0')
		fprintf(tmp, ".xf 1\n");
	if (ol_lev > 0)
		ol_lev--;
}

/*
 * Process a .ol command.
 */
do_ol()
{
        if (get_args() > 1)
		error("the .ol on line %d has an incorrect number of arguments", ln_cnt);
        if (lst_lev == MXLSTNST) {
                error("lists may nest only %d deep, the .ol on line %d was ignored", MXLSTNST, ln_cnt);
		return;
	}
	sp = &lst_stk[++lst_lev];
        ol_lev++;
        if (ol_lev == 1)
                sp->t_ind = RLIND;
        else
                sp->t_ind = LIND;
        sp->l_ind = 0;
        sp->l_lps = 2;
        if (ol_lev == 6)
                sp->l_typ = 3;
        else
                sp->l_typ = 1;
        if (ol_lev == 1)
                sp->label = "I";
        else if (ol_lev == 2)
                sp->label = "A";
        else if (ol_lev == 3)
                sp->label = "1";
        else if (ol_lev == 4)
                sp->label = "a";
        else if (ol_lev == 5)
                sp->label = "i";
        else
                sp->label = "1";
        if (*arg[1] == '\0')
                sp->le_sp = 1;
        else
                sp->le_sp = 0;
	init_lst();
}

/*
 * Process a .nr or .ds command.  The names and values of the registers
 * are stored in the hash table.  They are passed to nroff as well as
 * being processed.  They are processed so that
 * the value of number and string registers in section headings,
 * appendix headings and figure names can be substituted (since the
 * precise length of each of these must be known).  They are not
 * substituted into the regular text.
 */
do_reg(cmd)
{
	register int c;
	register int lastc = '\0';
	register int count = 0;  /* the number of characters in the name */
	int firstchar      = 1;  /* 1 for first character of the register value, 0 otherwise */
	char regbuf[3];
	char *reg = regbuf;
	char valbuf[100];  /* temporarily store the value of the register */
	char *val = valbuf;
	TBL *p;
	TBL *tp;

	while ((c=getc(infil)) == ' ')
		;
	while (c != ' ' && c != '\n') {
		*reg++ = c;
		count++;
		c = getc(infil);
	}
	if (count > 2) {
		error("the register name on line %d is too long", ln_cnt);
		reg -= count - 2;
	}
	*reg = '\0';
	if (c != '\n') {
	        while ((c=getc(infil)) == ' ')
		        ;
	        while (c != '\n') {
		        if (firstchar) {
				firstchar = 0;
	                        if (c == '"') {
					c = getc(infil);
					continue;
				}
		        }
                        if (c == boldc && lastc != '\\') {
                                *val++ = '\\';
                                *val++ = 'f';
                                if (bfont > 0)
                                        *val++ = 'P';
                                else
                                        *val++ = 'B';
                                bfont = -bfont;
                        } else if (c == italc && lastc != '\\') {
                                *val++ = '\\';
                                *val++ = 'f';
                                if (ifont > 0)
                                        *val++ = 'P';
                                else
                                        *val++ = 'I';
                                ifont = -ifont;
			} else if (c == '"' && lastc == '\\') {
				val--;
				while ((c=getc(infil)) != '\n')
					;
				break;
                        } else
                                *val++ = c;
			lastc = c;
			c = getc(infil);
		}
	}
	*val = '\0';
	if (cmd == DS) {
                fprintf(tmp, ".ds %s \"%s\n", regbuf, valbuf);  /* output the string definition */
                tp = tblinsrt(DSCHAR, regbuf);
	        tp->value.regvalue = strdup(valbuf);
	} else if (cmd == NR) {
		fprintf(tmp, ".nr %s %s\n", regbuf, valbuf);  /* output the number register definition */
		if ((p=tbllkup(NRCHAR, regbuf)) == NULL) {
	                tp = tblinsrt(NRCHAR, regbuf);
	                tp->value.regvalue = strdup(valbuf);
		} else
		        if (valbuf[0] == '+')
                                sprintf(p->value.regvalue, "%d", atoi(p->value.regvalue) + atoi(&valbuf[1]));
		        else if (valbuf[0] == '-')
                                sprintf(p->value.regvalue, "%d", atoi(p->value.regvalue) - atoi(&valbuf[1]));
			else
                                sprintf(p->value.regvalue, "%d", atoi(valbuf));
	}
}

/*
 * Process a .rl command.
 */
do_rl()
{
        if (get_args() > 1)
		error("the .rl on line %d has an incorrect number of arguments", ln_cnt);
        if (lst_lev == MXLSTNST) {
                error("lists may nest only %d deep, the .rl on line %d was ignored", MXLSTNST, ln_cnt);
		return;
	}
	sp = &lst_stk[++lst_lev];
        sp->t_ind = 6;
        sp->l_ind = 0;
        sp->l_lps = 2;
        sp->l_typ = 2;
        sp->label = "1";
        if (*arg[1] == '\0')
                sp->le_sp = 1;
        else
                sp->le_sp = 0;
	init_lst();
}

/*
 * Process a .sf command.
 */
do_sf()
{
	TBL *tp;
	char number[3];

	if (get_args() > 1)
		error("the .sf on line %d has an incorrect number of arguments", ln_cnt);
	printf(".ds %s %d\n", arg[1], figs[pos].count);
	tp = tblinsrt(DSCHAR, arg[1]);
	sprintf(number, "%d", figs[pos].count);
	tp->value.regvalue = strdup(number);
}

/*
 * Process a .sh command.
 */
do_sh()
{
	TBL *tp;
	char number[10];

	if (get_args() > 2)
		error("the .sh on line %d has an incorrect number of arguments", ln_cnt);
	if (chd_lev == 0)
		printf(".nr wk %d\n.ds %s \\n(wk\n", app_num, arg[1]);
        else {
		tp = tblinsrt(DSCHAR, arg[1]);
	        if (chd_lev == 1 && *arg[2] != '\0') {
		        printf(".ds %s %s%s\n", arg[1], hdnum, arg[2]);
			sprintf(number, "%s%s", hdnum, arg[2]);
			tp->value.regvalue = strdup(number);
	        } else {
		        printf(".ds %s %s\n", arg[1], hdnum);
			tp->value.regvalue = strdup(hdnum);
		}
	}
}

/*
 * Process a .sl command.
 */
do_sl()
{
	register int i = 1;   /* used as index to lst_stk */

	if (get_args() > 2)
		error("the .sl on line %d has an incorrect number of arguments", ln_cnt);
	printf(".ds %s\n", arg[1]);  /* make string empty */
	if (*arg[2] != '\0')
		while (lst_lev == 1 && i == 1 || lst_lev > 1 && i < lst_lev) {
			sp = &lst_stk[i++];
			if (sp->l_typ == 0 || sp->l_typ == 4)
				printf(".as %s %s%s\n", arg[1], sp->label, arg[2]);
			else
				printf(".af uh %s\n.nr uh %d\n.as %s \\n(uh%s\n", sp->label, sp->itm_num, arg[1], arg[2]);
                }
        if (*arg[2] != '\0' && lst_lev > 1 || *arg[2] == '\0') {
                sp = &lst_stk[lst_lev];
                if (sp->l_typ == 0 || sp->l_typ == 4)
                        printf(".as %s %s\n", arg[1], sp->label);
                else
			printf(".af uh %s\n.nr uh %d\n.as %s \\n(uh\n", sp->label, sp->itm_num, arg[1], arg[2]);
        }
}

/*
 * Process a .so command.
 */
do_so()
{
	FILE *tinfil;

	if (get_args() > 1)
		error("the .so on line %d has an incorrect number of arguments", ln_cnt);
	if ((tinfil = fopen(arg[1], "r")) != NULL) {
	        *fsp++ = infil;
	        *lsp++ = ln_cnt;
		*++fnp = strdup(arg[1]);
		infil = tinfil;
	        ln_cnt = 0;
	} else
		error("cannot open the file %s on line %d", arg[1], ln_cnt);
}

/*
 * Process a .vl command.
 */
do_vl()
{
        if (get_args() > 2)
		error("the .so on line %d has an incorrect number of arguments", ln_cnt);
        if (lst_lev == MXLSTNST) {
                error("lists may nest only %d deep, the .vl on line %d was ignored", MXLSTNST, ln_cnt);
		return;
	}
	sp = &lst_stk[++lst_lev];
        if (*arg[1] == '\0' || strcmp(arg[1], "\\&") == 0)
                sp->t_ind = LIND;
        else
                sp->t_ind = atoi(arg[1]);
        sp->l_ind = 0;
        sp->l_lps = 0;
        sp->l_typ = 4;
        sp->label = "\0";
        if (*arg[2] == '\0')
                sp->le_sp = 1;
        else
                sp->le_sp = 0;
	init_lst();
}

/*
 * Put the commands that are going to be pre-processed into
 * the hash table.
 */
entr_cmd()
{
	TBL *tp;

        tp = tblinsrt(CMDCHAR, "al");   tp->value.cmdnum = AL;
        tp = tblinsrt(CMDCHAR, "ap");   tp->value.cmdnum = AP;
        tp = tblinsrt(CMDCHAR, "bl");   tp->value.cmdnum = BL;
        tp = tblinsrt(CMDCHAR, "cm");   tp->value.cmdnum = CM;
        tp = tblinsrt(CMDCHAR, "de");   tp->value.cmdnum = DE;
        tp = tblinsrt(CMDCHAR, "dl");   tp->value.cmdnum = DL;
        tp = tblinsrt(CMDCHAR, "ds");   tp->value.cmdnum = DS;
        tp = tblinsrt(CMDCHAR, "fg");   tp->value.cmdnum = FG;
        tp = tblinsrt(CMDCHAR, "fo");   tp->value.cmdnum = FO;
        tp = tblinsrt(CMDCHAR, "gf");   tp->value.cmdnum = GF;
        tp = tblinsrt(CMDCHAR, "hd");   tp->value.cmdnum = HD;
        tp = tblinsrt(CMDCHAR, "il");   tp->value.cmdnum = IL;
        tp = tblinsrt(CMDCHAR, "le");   tp->value.cmdnum = LE;
        tp = tblinsrt(CMDCHAR, "lx");   tp->value.cmdnum = LX;
        tp = tblinsrt(CMDCHAR, "nr");   tp->value.cmdnum = NR;
        tp = tblinsrt(CMDCHAR, "ol");   tp->value.cmdnum = OL;
        tp = tblinsrt(CMDCHAR, "rl");   tp->value.cmdnum = RL;
        tp = tblinsrt(CMDCHAR, "sf");   tp->value.cmdnum = SF;
        tp = tblinsrt(CMDCHAR, "sh");   tp->value.cmdnum = SH;
        tp = tblinsrt(CMDCHAR, "sl");   tp->value.cmdnum = SL;
        tp = tblinsrt(CMDCHAR, "so");   tp->value.cmdnum = SO;
        tp = tblinsrt(CMDCHAR, "vl");   tp->value.cmdnum = VL;
}

/*
 * Put the pre-defined strings in the hash table.
 */
entr_str()
{
	struct tm *t;
	struct tm *localtime();
	char s[25];
	char mon[10];
	int hour;
	char aorpm[5];
	TBL *tp;

	tp = tblinsrt(DSCHAR, "A");
	tp->value.regvalue = "\\fBAu\\fR";

	tp = tblinsrt(DSCHAR, "U");
	tp->value.regvalue = "UNIX";

	tp = tblinsrt(DSCHAR, "sd");
	t = localtime(time());
	sprintf(s, "%d/%d/%d", t->tm_mon+1, t->tm_mday, t->tm_year);
	tp->value.regvalue = strdup(s);

	tp = tblinsrt(DSCHAR, "fd");
	switch (t->tm_mon) {
	case 0 : strcpy(mon, "January");   break;
	case 1 : strcpy(mon, "February");  break;
	case 2 : strcpy(mon, "March");     break;
	case 3 : strcpy(mon, "April");     break;
	case 4 : strcpy(mon, "May");       break;
	case 5 : strcpy(mon, "June");      break;
	case 6 : strcpy(mon, "July");      break;
	case 7 : strcpy(mon, "August");    break;
	case 8 : strcpy(mon, "September"); break;
	case 9 : strcpy(mon, "October");   break;
	case 10: strcpy(mon, "November");  break;
	case 11: strcpy(mon, "December");  break;
	}
	sprintf(s, "%s %d, 19%d", mon, t->tm_mday, t->tm_year);
	tp->value.regvalue = strdup(s);

	tp = tblinsrt(DSCHAR, "dw");
	switch (t->tm_wday) {
	case 0: strcpy(s, "Sunday");     break;
	case 1: strcpy(s, "Monday");     break;
	case 2: strcpy(s, "Tuesday");    break;
	case 3: strcpy(s, "Wednesday");  break;
	case 4: strcpy(s, "Thursday");   break;
	case 5: strcpy(s, "Friday");     break;
	case 6: strcpy(s, "Saturday");   break;
	}
	tp->value.regvalue = strdup(s);

	tp = tblinsrt(DSCHAR, "tw");
	sprintf(s, "%d:%d", t->tm_hour, t->tm_min);
	tp->value.regvalue = strdup(s);

	tp = tblinsrt(DSCHAR, "nt");
	if (t->tm_hour > 12) {
		hour = t->tm_hour - 12;
		strcpy(aorpm, "p.m.");
	} else if (t->tm_hour == 0) {
		hour = 12;
		strcpy(aorpm, "a.m.");
	} else {
		hour = t->tm_hour;
		strcpy(aorpm, "a.m.");
	}
	sprintf(s, "%d:%d %s", hour, t->tm_min, aorpm);
	tp->value.regvalue = strdup(s);
}

/*
 * Warning messages are output to standard error using this function.
 */
/*VARARGS*/
error(s)
char *s;
{
	if (infil != stdin)
	        fprintf(stderr, "%s: ", *fnp);
	fprintf(stderr, "%r", &s);
	fprintf(stderr, "\n");
	exit(1);
}

/*
 * A user defined macro has been recognized.  Get the macro definition
 * (pointed to by t) and put it into a memory file.  Push this file
 * onto the input file stack so it is read next.  Change the inmac
 * variable to 1 so that the right things are done with the filename
 * stack and ln_cnt.  If a \$ is found, substitute the appropriate
 * argument.
 */
get_mac(t)
TBL *t;
{
	register char *p;
	register char *ap = '\0';
	int argnum;
	FILE *detmp;

	get_args();
	inmac = 1;
	detmp = memopen("de temp file", "w");
        for (p=t->value.regvalue; *p != '\0'; p++) {
                if (*ap != '\\' && *p == '\\' && *(p+1) == '$') {
                        argnum = *(p+=2) - '0';
			fputs(arg[argnum], detmp);
                } else
                        putc(*p, detmp);
		ap = p;
	}
	*fsp++ = infil;
	memreopen(detmp, "r");
	infil = detmp;
}

/*
 * Output the commands common to all list initializations.
 */
init_lst()
{
        if (sp->l_typ > 0 && sp->l_typ < 4)
                fprintf(tmp, ".nr l%d 0\n.af l%d %s\n", lst_lev, lst_lev, sp->label);
	sp->itm_num = 0;
        fprintf(tmp, ".nr i%d (\\n(.i/\\w\\| \\|\)+%d\n", lst_lev, sp->t_ind);
        if (sp->l_lps == 0)
                fprintf(tmp, ".nr j%d (\\n(.i/\\w\\| \\|\)+%d\n", lst_lev, sp->l_ind);
}

/*
 * Return a pointer to a string which is a section number plus
 * the appropriate number of blanks.  hdnum is a pointer to the
 * section number and hdlen is the length of the string to be
 * returned.
 */
char *
pd_sec_hd(hdnum, hdlen)
char *hdnum;
{
	register int i;
	char buf[1000];
	register char *p;

	strcpy(buf, hdnum);
	p = &buf[strlen(buf)];
	for (i=strlen(buf); i<hdlen; i++) {
		*p++ = '\\';
		*p++ = ' ';
	}
	*p = '\0';
	return(strdup(buf));
}

/*
 * Insert a command name, a macro name, a number register name or a
 * string register name into the hash table.  type is an ascii
 * character (CMDCHAR, DECHAR, NRCHAR or DSCHAR)
 * which represents the type of item being entered and name
 * is its name.  It returns a pointer to the entry in the hash table.
 * The calling functions then insert the value of the name into the
 * hash table.
 */
TBL *
tblinsrt(type, name)
char type;
char *name;
{
	register unsigned h;
	TBL *s;
	TBL *t;

        s = (TBL *) malloc(sizeof(TBL));
	s->name[0] = type;
	strcpy(&(s->name[1]), name);
	h = hash(type, name);
	t = hashp[h];
	hashp[h] = s;
	s->chain = t;
	return(s);
}

/*
 * Look up a command, macro, number register or string register in the
 * hash table.  Return a pointer to the hash table entry or NULL if it
 * isn't found.
 */
TBL *
tbllkup(type, name)
char type;
char *name;
{
	register unsigned h;
	TBL *s;

	h = hash(type, name);
	for (s = hashp[h]; s!= NULL; s = s->chain)
		if (type == s->name[0] && strcmp(name, &(s->name[1])) == 0)
			return(s);
	return(NULL);
}

/*
 * Make an entry for the table of contents or any of the figure
 * tables.  The proper .dm command has already been given and
 * a .sp if appropriate.  indent is the amount to indent a
 * heading if it is more than one line, narg is the argument
 * that begins the text of the entry (2 for .hd, 1 for .ap),
 * arg_cnt is the number of arguments in the entry.  cmd is
 * needed because of the too long error message.  If cmd is FG
 * the argument number is one less than the number that would
 * otherwise come out in the error message.  Strings
 * and number registers will mess up the table of contents entry
 * under two circumstances: 1) the register was never defined,
 * 2) the string has been defined using .sf, .sh or .sl.
 */
tc_entry(indent, narg, arg_cnt, cmd)
{
	register int i, j;
	char buf[150];           /* holds a line of the table of contents */
	register char *p;        /* pointer to buf */
	int arg_len[MAXARGS];    /* stores the length of each word */
	int line = lin_len - 3;  /* the "line length" for the table of contents, the 3 spaces are for the page number */
	int fullflg;             /* 1 when a line is full, 0 when not */
	int doneflg = 0;         /* 1 when an entire entry has been formatted */
	int filled;              /* the number of spaces on a line that have been filled */

	for (i=1; i<=arg_cnt; i++) {
		arg_len[i] = strlen(arg[i]);
		if (indent + arg_len[i] + 1 > line) {
			if (cmd == FG)
			        error("argument %d on line %d is too long", i-1, ln_cnt);
			else
			        error("argument %d on line %d is too long", i, ln_cnt);
			arg_len[i] = line - indent - 1;  /* phoney up the arg_len so that the program can continue even though */
                                                         /* the table of contents entry will look awful */
		}
	}
        for (i=1; i<=arg_cnt; i++)
                for (j=0; j<strlen(arg[i])-1; j++)
                        if (arg[i][j] == '\\')  /* adjust for escape sequences */
                                switch (arg[i][++j]) {
                                case '\\':  arg_len[i] -= 1;  break;
                                case 'e' :  arg_len[i] -= 1;  break;
                                case '&' :  arg_len[i] -= 2;  break;
                                case ' ' :  arg_len[i] -= 1;  break;
                                case '(' :  arg_len[i] -= 3;  break;  /* this assumes each special character is one character wide */
                                case 'f' :  arg_len[i] -= 3;  break;
                                case '%' :  arg_len[i] -= 2;  break;
				default  :  arg_len[i] -= 1;  break;  /* assume that other escaped characters are one character wide */
                                }
	while (!doneflg) {
	        p = buf;
	        *p = '\0';
		filled = indent;
		fullflg = 0;
                while (!fullflg && !doneflg) {
			if (*arg[narg] == '\0' || strcmp(arg[1], "\\&") == 0)
				narg++;
			else if (filled + arg_len[narg] + 1 <= line) {
                                strcat(p, arg[narg]);
                                strcat(p, " ");
                                filled += arg_len[narg++] + 1;
                        } else
                                fullflg = 1;
                        if (narg > arg_cnt)
                                doneflg = 1;
	        }
                if (!doneflg)
                        fprintf(tmp, "%s\n.br\n", p);
	}
        strcat(p, "\\fR\\&");  /* if the entry was in bold the font changes to roman after the entry and before the dots */
	if (filled % 2 == 1 && filled < line) {  /* add a blank to get .'s started in right column */
		strcat(p, " ");
		filled++;
	}
	while (filled < line) {  /* add .'s */
		strcat(p, ". ");
		filled += 2;
	}
        fprintf(tmp, "%s\\c\n.if \\n%%<100 \\ \\c\n.if\\n%%<10 \\ \\c\n\\n%%\n.br\n.di\n", p);
}

past_eof()
{
	error ("attempt to read past end-of-file, due to garbage input");
	exit(1);
}
